% NbrSpAnalyze   Point Cloud Spatial Analysis
%
% File: spanalize.m
%
% Description:
%    Perform local feature attribution via eigenspace analysis.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [classes] = spanalyse(queries, database, ...)
%
% Inputs:
%    queries  - .
%    database - .
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    classes - Structure containing the spatial analysis classifications.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    None.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-JAN-28   K. Damkjer
%         Initial Coding.
%      2013-FEB-04   K. Damkjer
%         Change output to a struct for easier referencing.
%

%******************************************************************************
% spanalyze
%******************************************************************************
function [ classes ] = nbrspanalyze( points, neighborhoods, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    queries  - .
   %
   %    database - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %

   userParams = parseInputs(varargin{:}); % Parse Inputs

%   maketimebar = false;
%   
%   if (~isempty(varargin) && strcmp('timebar',varargin{1}))
%      maketimebar = true;
%   end
   
   % Sizing metrics
   dimensions=size(points, 1);
   elements=size(neighborhoods, 2);

   % Pre-allocate classes
   norms=zeros(dimensions, elements);
   feats=zeros(dimensions, elements);
   dists2means=zeros(1,elements);
   ints=zeros(1,elements);

   if (~isempty(userParams.test))
      if (length(neighborhoods) ~= length(userParams.test))
         error('Damkjer:nbrspanalyze:size',                                 ...
               ['When test sets are provided, '                             ...
                'there must be one for each point set.']);
      end
      
      neighborhoods=cellfun(@(x,y) [x;y], userParams.test, neighborhoods,   ...
                            'UniformOutput', false);
%      neighborhoods=cellfun(@(x,y) [x(1);y], userParams.test, neighborhoods,   ...
%                            'UniformOutput', false);
   end

   % This process may take a while. Display time bar while processing.
   if (userParams.maketimebar)
      msg='Analyzing Neighborhoods...';
      tstart=tic;
      h = timebar(1, elements, msg, tstart);
   end
   
   % Step through the queries in chunks.
   step=1000;

   if (userParams.maketimebar)
      tic;
   end

   for elem=1:step:elements

      last=min(elem+step-1,elements);

      NN=neighborhoods(elem:last);
      
      cells=cell(1,length(NN));

      for n=1:length(NN)
         cells{n}=points(:,NN{n})';
      end
      
      [covs,dist2mean,inty]=fastcov(cells);

%       points(:,1:10)
%       NN{1:10}
%       cells{1:10}
%       covs{1:10}
%       dist2mean{1:10}
%       inty{1:10}
%          
%       error('stop');
         
      % Compute the neighborhood covariance eigenvalues.
      [V,D]=par_eig(covs);
      
      for nbr=1:length(NN)

         % skip underconstrained neighborhoods (possible with radius searches)
         if (length(NN{nbr})<5 || any(any(isnan(covs{nbr}))))
            continue;
         end

         [~,index]=min(abs(D{nbr}));
         A=V{nbr};
         norms(:,elem+nbr-1)=abs(A(:,index));
         
         % Option 1: Define features as eigenvalues
         % feats(:,elem+nbr-1)=sort(D{nbr},'descend');
         
         % Option 2: Define features as singular values. Eigenvalues are not
         % guaranteed to be in any order. We want them in descending order.
         %
         % max is to get rid of pesky negative zeros.
         feats(:,elem+nbr-1)=sqrt(max(sort(D{nbr},'descend'),0));

         dists2means(elem+nbr-1)=dist2mean{nbr};
         ints(elem+nbr-1)=inty{nbr};
      end

      % Update the time bar.
      if (userParams.maketimebar && toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (userParams.maketimebar && all(ishghandle(h, 'figure')))
      close(h);
   end

   % Compute Normalized Features
   energy=sum(feats,1);
   nfeats=bsxfun(@times,feats.',1./energy.').';
   nfeats(isnan(nfeats))=1;
   nfeats(nfeats==0)=1;
   
   % Compute Omnivariance
%   omnivar=prod(feats,1).^(1/dimensions);
   omnivar=prod(feats.*feats,1).^(1/dimensions);
%   omnivar=prod(nfeats,1).^(1/dimensions);
   
   % Compute Eigen Entropy
   entropy=-sum(nfeats.*log(nfeats))./log(dimensions);
%   entropy(entropy==1)=0;
   entropy(isnan(entropy))=0;
   
   % Compute Eigen Structure
   structure=1-entropy;
   
   % Compute Fractional Anisotropy
   evmeans=mean(feats,1);
   evdevs=feats-repmat(evmeans,dimensions,1);
   numer=dimensions.*sum(evdevs.^2,1);
   denom=(dimensions-1).*sum(feats.^2,1);
   fa=(numer./denom).^(1/2);
   
   % Compute Anisotropy
   ani=(feats(1,:)-feats(dimensions,:))./(feats(1,:));
   
   % Compute Isotropy
   iso=1-ani;
   
   % Compute dimensional degree
   dims=zeros(size(feats));
   dims(1:dimensions-1,:)=(feats(1:dimensions-1,:)-feats(2:dimensions,:))./...
                           repmat(feats(1,:),dimensions-1,1);
   dims(dimensions,:)=iso;
   
   % Compute Dimensional Embedding
   [~,label]=max(dims,[],1);

   % Compute dimensional entropy
%   alpha=[dims;iso];
   alpha=dims;
   alpha(isnan(alpha))=1;
   alpha(alpha==0)=1;
   de=-sum(alpha.*log(alpha))./log(dimensions);
   
%    if (~isreal(de))
%       disp('de...');
%       disp(de);
%       disp('feats...');
%       disp(feats);
%       disp('eigs...');
%       disp(feats.*feats);
%    end
   
   % Populate feature classes
   classes.features=feats;
   classes.normals=norms;
   classes.dimensionality=dims;
   classes.isotropy=iso;
   classes.anisotropy=ani;
   classes.FA=fa;
   classes.entropy=entropy;
   classes.structure=structure;
   classes.omnivariance=omnivar;
   classes.labeling=label;
   classes.dists2means=dists2means;
   classes.intensity=ints;
   classes.de=de;

end

%******************************************************************************
% parseInputs
%******************************************************************************
function [userParams] = parseInputs(varargin)
   % Support function to parse inputs into userParams structure
   %
   % Parameters:
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    userParams - .

   userParams = struct('maketimebar', false, 'test', []);
   
   % Parse the Property/Value pairs
%   if rem(length(varargin), 2) ~= 0
%      error('Damkjer:PropertyValueNotPair', ...
%            ['Additional arguments must take the form of Property/Value '...
%             'pairs']);
%   end
   
   propertyNames = {'timebar', 'test'};

   while ~isempty(varargin)
      
      if islogical(varargin{1})
         userParams.maketimebar = varargin{1};
         varargin(1) = [];
      elseif iscell(varargin{1})
         userParams.test = varargin{1};
         varargin(1) = [];
      elseif (ischar(varargin{1}) &&                                        ...
              (length(varargin) < 2 || ischar(varargin{2})) &&              ...
              strcmp('timebar',varargin{1}))
         userParams.maketimebar = true;
         varargin(1) = [];
      else
         property = varargin{1};
         value    = varargin{2};

         % If the property has been supplied in a shortened form, lengthen it
         iProperty = find(strncmpi(property, propertyNames, length(property)));

         if isempty(iProperty)
            error('Damkjer:InvalidProperty', 'Invalid Property');
         elseif length(iProperty) > 1
            error('Damkjer:AmbiguousProperty',                              ...
                  'Supplied shortened property name is ambiguous');
         end

         property = propertyNames{iProperty};
      
         switch property
         case 'timebar'
            if (islogical(value))
             userParams.maketimebar = value;
            else
               error('Damkjer:InvalidCount',                                ...
                     'Timebar must be called with a logical parameter');
            end
         case 'test'
            if (isnumeric(value))
               userParams.test = fix(value);
            else
               error('Damkjer:InvalidRadius',                               ...
                     'Test must be called with a set of indices');
            end
         end

         varargin(1:2) = [];
      end
   end
end
