% Thin_Dyn   Point Cloud Thinning (Dyn et al)
%
% File: thin_Dyn.m
%
% Description:
%    Perform thinning according to Dyn 2004.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [results] = spanalyse(points, threshold, ...)
%
% Inputs:
%    points - .
%    threshold - must be positive. between 0-1, treated as percentage, > 1
%                points.
%
%    Option strings may be provided as abbreviations as long as they resolve to
%    a unique option.
%
%    'neighbors' - .
%    'radius'    - .
%    'counts'    - .
%    'steps'     - .
%
% Outputs:
%    results - Thinned point cloud.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    Dyn reference.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-DEC-28  Initial coding.
%

%******************************************************************************
% thin_Dyn
%******************************************************************************
function [ results ] = thin_Damkjer( points, threshold, varargin )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    points  - .
   %
   %    threshold - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %
   
   userParams = parseInputs(varargin{:}); % Parse Inputs

   %***
   % CONFIRM UNIQUENESS
   %***
   
   % Make sure that all points in the set are unique...
   disp('Culling duplicate points...');
   tstart=tic;
   check = unique(points.','rows').';
   disp(['...done in ' num2str(toc(tstart)) 's']);

   if (any(size(check) ~= size(points)))
      error('Points must be unique. Try: unique(X.'',''rows'',''stable'').''');
   end
   
   %***
   % INDEX POINT SET
   %***
   disp('Indexing points...');
   tstart=tic;
   database = VpTree(points(1:3,:));
   queries = points;
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   %***
   % CREATE NEIGHBORHOODS
   %***

   database.excludeWithin(0.000000001); % Set um precision on searches.
   
   % Sizing metrics
   dimensions=size(queries, 1);
   elements=size(queries, 2);
   
   nbrs  = cell(1,elements);
   dists = cell(1,elements);
   test  = num2cell(1:elements, 1);

   % This process may take a while. Display time bar while processing.
   msg='Creating Neigborhoods...';
   tstart=tic;
   h = timebar(1, elements, msg, tstart);

   disp(msg);

   % Step through the queries in chunks.
   step=1000;

   tic;

   for elem=1:step:elements

      % The last available element may be closer than elem + step.
      last=min(elem+step-1,elements);
      
      % Get the nearest neighbors of elem
      if (userParams.radius > 0 && userParams.neighbors <= 0)
         % Perform a fixed radius search
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.rnn(queries(1:3,elem:last),          ...
                                           userParams.radius);
      elseif (userParams.radius <= 0 && userParams.neighbors > 0)
         % Search unconstrained neighbors
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.knn(queries(1:3,elem:last),          ...
                                           userParams.neighbors);
      elseif (userParams.radius > 0 && userParams.neighbors > 0)
         % Search constrained to radius
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.knn(queries(1:3,elem:last),          ...
                                           userParams.neighbors,            ...
                                           'lim', userParams.radius);
      elseif (~isempty(userParams.counts) && userParams.radius <= 0)
         % Search unconstrained neighbors
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.knn(queries(1:3,elem:last),          ...
                                           max(userParams.counts));
      elseif (~isempty(userParams.counts) && userParams.radius > 0)
         % Search constrained to radius
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.knn(queries(1:3,elem:last),          ...
                                           max(userParams.counts),          ...
                                           'lim', userParams.radius);
      elseif (~isempty(userParams.steps))
         % Perform a fixed radius search
         [nbrs(elem:last),                                                  ...
          dists(elem:last)] = database.rnn(queries(1:3,elem:last),          ...
                                           max(userParams.steps));
      end

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end

   disp(['...done in ' num2str(toc(tstart)) 's']);

   %***
   % COMPUTE DUAL NEIGHBORHOODS
   %***

   disp('Computing Dual Neigborhoods...');
   tstart=tic;
   duals=fastsetdual(nbrs);
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   %***
   % COMPUTE NATIVE ATTRIBUTES
   %***
   disp('Computing Native Attributes...');
   tstart=tic;

   natives = nbrspanalyze(points, nbrs, test, 'timebar');
%   natives = nbrspanalyze(points, nbrs, 'timebar');

%    if (any(isnan(natives.entropy)))
%       disp('Ugh... seriously?');
%    end
%    
   disp(['...done in ' num2str(toc(tstart)) 's']);

   %***
   % COMPUTE NEIGHBORHOOD ATTRIBUTES
   %***
   disp('Computing Neighborhood Attributes...');
   tstart=tic;
   
   feats = nbrspanalyze(points, nbrs, 'timebar');

%    if (any(isnan(feats.entropy)))
%       disp('Ugh... seriously?');
%    end
%    
   disp(['...done in ' num2str(toc(tstart)) 's']);
   
   %***
   % COMPUTE SIGNIFICANCES
   %***
   sigs=zeros(1,elements);

   msg='Computing Significances...';
   
   disp(msg);
   
   tstart=tic;
   h = timebar(1, elements, msg, tstart);

   tic;

   for elem=1:elements
      sigs(elem)=abs(feats.entropy(elem)-natives.entropy(elem));

      % Update the time bar.
      if (toc > 1)
         tic;
         h = timebar(elem, elements, msg, tstart, h);
      end
   end

   % Close the time bar, if still open.
   if (all(ishghandle(h, 'figure')))
      close(h);
   end

%    if (any(isnan(sigs)))
%       disp('Ugh... seriously?');
%    end
%
   disp(['...done in ' num2str(toc(tstart)) 's']);

%    origFormat = get(0, 'format');
%    format('longG');
%    min(sigs)
%    max(sigs)
%    set(0, 'format', origFormat);

%    results = natives;
% end
%   
% function cont
%    error('Stop');
   

   %***
   % PARTITION THE DATA SET
   %***

%   disp('Partition by Label...');
%   tstart=tic;
%
%   parts = cell(1,dimensions);
%
%   for dim=1:dimensions
%      parts{dim}=sort(uint64(find(feats.labeling==dim)),'ascend');
%   end
%
%   [~,idx]=sort(cellfun('length',parts),'descend');
%   parts=parts(idx);
%   
%   disp(['...done in ' num2str(toc(tstart)) 's']);

   %***
   % THIN BY LABEL
   %***

%   fraction=0;

   removed=zeros(1,elements-threshold);
   dual_size=zeros(1,elements-threshold);
   current=1;

%   removed_wgts=cell(1,3);
   fellback = 0;

%   for dim=1:dimensions
%      desired  = threshold * length(parts{dim}) / elements + fraction;
%      tau      = floor(desired);
%      fraction = desired-tau;
%      heap     = SplayTree(sigs(parts{dim}),parts{dim});
%      n        = length(parts{dim})-tau;
      
%      removed_wgts{dim}=zeros(1,n);

      tau=threshold;
      heap = SplayTree(sigs,1:length(sigs));
      n = length(sigs) - tau;
      removed_wgts=zeros(1,n);

%      msg=['Thin Dimension ' num2str(dim) '(' num2str(length(parts{dim})) ...
%                                          '->' num2str(tau) ')...'];
      msg='Thin...';
      tstart=tic;
      h = timebar(1, n, msg, tstart);

      disp(msg);
      tic;

%       disp(nbrs{318935})
%       disp(test{318935})
%       disp(duals{318935})
% 
%       disp(nbrs{458702})
%       disp(test{458702})
%       disp(duals{458702})

      for x=1:n
         
%         disp('Pop');
         
%         heap_size = heap.size();

         % Locate the least significant point to remove
         [wgt, idx]=heap.pop_head();

%         disp('Popped');
         
%         if (heap.size() ~= (heap_size - 1))
%            disp(['...and there it is: ' num2str(heap_size) ' - 1 vs. ' num2str(heap.size())]);
%            disp(idx);
%            error('oops');
%         end
         
%         if (isempty(nbrs{idx}) || isempty(duals{idx}))
%            disp(idx);
%         end

%         if (isempty(nbrs{idx}) && isempty(duals{idx}))
%            disp('uh oh');
%            disp(idx);
%         end
%         idx
%         wgt

%          if (any(removed==idx))
%             disp('uh oh');
%             disp(['Index already removed ' num2str(idx)]);
%          end

         % Mark the point as removed
         removed(current)=idx;
%         removed_wgts{dim}(x)=wgt;
         removed_wgts(x)=wgt;


         % Distribute the points in the test set to the neighbor's test sets.
         neighbors = nbrs{idx};
         samples   = test{idx};
         
%         neighbors
%         samples
         
         for samp=1:length(samples)
            % Find the closest
%            [dsq,new_idx]=min(sum((bsxfun(@minus, points(:,neighbors), ...
%                                          points(:,samples(samp)))).^2, 1));

            [dsq,new_idx]=min(sum((bsxfun(@minus, points(1:3,neighbors), ...
                                          points(1:3,samples(samp)))).^2, 1));

%             samples(samp)
%             points(1:3,neighbors)
%             points(1:3,samples(samp))
%             dsq
%             new_idx
%             neighbors(new_idx)

            % Update the test set
            test{neighbors(new_idx)}(end + 1,1) = samples(samp);

%             if (samples(samp) == 458702)
%                disp(['relocating 458702 from test{' num2str(idx) '} to test{'...
%                      num2str(neighbors(new_idx)) '}']);
%                
%                disp(test{neighbors(new_idx)});
%             end
% 
%             if (neighbors(new_idx) == 458702)
%                disp(['relocating ' num2str(samples(samp)) ' from test{' num2str(idx) '} to test{458702}']);
%                
%                disp(test{neighbors(new_idx)});
%             end

%            tdist(neighbors(new_idx)) = max(tdist(neighbors(new_idx)),...
%                                            sqrt(dsq));
         end

         % The removed element neighborhood is eliminated, so it no longer
         % contains the neighbors. Update the neighbor duals appropriately.
         for nbr=1:length(neighbors)
%             if (neighbors(nbr) == 458702)
%                disp([num2str(idx) '''s neighborhood no longer contains 458702']);
%                disp(duals{neighbors(nbr)});
%             end
            
            duals{neighbors(nbr)}=duals{neighbors(nbr)}(duals{neighbors(nbr)}~=idx);

%             if (neighbors(nbr) == 458702)
%                disp(duals{neighbors(nbr)});
%             end
         end

         % Find neighborhoods that contain removed element
         neighborhoods=duals{idx};
         dual_size(current)=length(neighborhoods);
         neighborhoods=sort(neighborhoods(neighborhoods~=idx), 'ascend');

         % The fallback neighborhood is identical for all neighborhoods, but
         % may be expensive to compute. Allow it to be lazily instantiated.
         fallback_pool = [];
         
         % Replace removed element in each neighborhood with nearest neighbor's
         % neighbor
         for hood=1:length(neighborhoods)

            % Build up the list of potential new neighbors
%            disp('building candidates...');

            nbr_pool=fastsetunion({nbrs{nbrs{neighborhoods(hood)}}});
            non_nbrs=fastsetunion({nbrs{neighborhoods(hood)},idx,neighborhoods(hood)});
            nbr_pool=nbr_pool(~ismembc(nbr_pool(:), non_nbrs(:)));

            if (isempty(nbr_pool))
               if (isempty(fallback_pool))
                  fallback_pool = uint64(find(~ismember(1:size(points,2),removed)));
               end
               fellback = fellback + 1;
               nbr_pool = fallback_pool;
               nbr_pool=nbr_pool(~ismembc(nbr_pool(:), non_nbrs(:)));
            end

            % Find the closest
%            disp('finding closest...');

%% Obsolete code
%            % Don't use VP Tree for single nearest neighbor... overhead not
%            % worth it.
%            locdb = VpTree(points(:,nbr_pool));
%            nn=locdb.knn(points(:,idx),1);
%            new_idx=nn{1}
%%
%             points(1:3,nbr_pool)
%             points(1:3,neighborhoods(hood))
%             
            [dsq,new_idx]=min(sum((bsxfun(@minus, points(1:3,nbr_pool), ...
                                          points(1:3,neighborhoods(hood)))).^2, 1));

%             points(1:3,nbr_pool)
%             points(1:3,neighborhoods(hood))
%             bsxfun(@minus, points(1:3,nbr_pool), points(1:3,neighborhoods(hood)))
%             (bsxfun(@minus, points(1:3,nbr_pool), points(1:3,neighborhoods(hood)))).^2
%             sum((bsxfun(@minus, points(1:3,nbr_pool), points(1:3,neighborhoods(hood)))).^2,1)
%             [dsq,new_idx]=min(sum((bsxfun(@minus, points(1:3,nbr_pool), points(1:3,neighborhoods(hood)))).^2,1))

            % Update the neighborhood
%            disp('updating neighborhood...'); 

            new_dist = sqrt(dsq);

%             if (neighborhoods(hood) == 458702)
%                disp(['replace ' num2str(idx) ' in 458702''s neighborhood']);
%                disp('Pool...');
%                disp(nbr_pool(new_idx));
%                disp('Before...');
%                disp(nbrs{neighborhoods(hood)});
%             end
%             
%             if (nbr_pool(new_idx) == 458702)
%                disp(['458702 is replacing ' idx ' in ' neighborhoods(hood) '''s neighborhood']);
%                disp(nbrs{neighborhoods(hood)});               
%             end
%             
            % Maintain a partially sorted list of indices by max distance.
            if (new_dist > dists{neighborhoods(hood)}(end))
               dists{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx) = ...
                  dists{neighborhoods(hood)}(end);
               
               nbrs{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx) =  ...
                  nbrs{neighborhoods(hood)}(end);

               dists{neighborhoods(hood)}(end) = sqrt(dsq);
               
               nbrs{neighborhoods(hood)}(end)  = nbr_pool(new_idx);
            else
               dists{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx) = ...
                  sqrt(dsq);
               
               nbrs{neighborhoods(hood)}(nbrs{neighborhoods(hood)}==idx) =  ...
                  nbr_pool(new_idx);
            end
            
            if (isempty(nbrs{neighborhoods(hood)}))
               disp(neighborhoods(hood));
            end
%            nbrs{neighborhoods(hood)}
%            dists{neighborhoods(hood)}
            
            % Update the dual
%            disp('updating dual...');


%            duals{nbr_pool(new_idx)}
            duals{nbr_pool(new_idx)}(end + 1,1) = neighborhoods(hood);
%            duals{nbr_pool(new_idx)}
%             disp(nbrs{neighborhoods(hood)});
%             disp(duals{nbr_pool(new_idx)});

%             error('stop');
%             if (neighborhoods(hood) == 458702)
%                disp('After...');
%                disp(nbrs{neighborhoods(hood)});
%                disp([num2str(nbr_pool(new_idx)) ' is now in 458702''s neighborhood']);
%                disp(duals{nbr_pool(new_idx)});
%             end
% 
%             if (nbr_pool(new_idx) == 458702)
%                disp(nbrs{neighborhoods(hood)});               
%                disp(['458702 is now in ' num2str(nbr_pool(new_idx)) '''s neighborhood']);
%                disp(duals{nbr_pool(new_idx)});
%             end

         end
         
         nbrs{idx}=[];
         duals{idx}=[];
         test{idx}=[];
         
         % Update analysis for elements
%         disp('updating analysis...');

%          if (isempty(neighborhoods))
%             warning('Damkjer:strange','empty neighborhoods');
%          end
%          
%          if (isempty(neighbors))
%             warning('Damkjer:stranger','empty neighbors');
%          end
%          
         if (isempty(neighborhoods))
            neighborhoods=neighbors;
         end
         
%         if (~isempty(neighborhoods) && ~isempty(neighbors))
         neighborhoods=fastsetunion({neighborhoods, neighbors});
%         elseif (isempty(neighborhoods) && ~isempty(neighbors))
%            neighborhoods=neighbors;
%         end
         
%         nbrs{neighborhoods}
%         test{neighborhoods}
         
%         error('Stop');

         if (~isempty(neighborhoods))
            locfeats = nbrspanalyze(points,                                 ...
                                    nbrs(neighborhoods));
%            locfeats = nbrspanalyze(points,                                 ...
%                                    nbrs(neighborhoods),                    ...
%                                    test(neighborhoods));

            locsigs=zeros(1,length(neighborhoods));

            for hood=1:length(neighborhoods)
                locsigs(hood)=max(abs(natives.entropy(test{neighborhoods(hood)})-locfeats.entropy(hood)));
            end

         % Update heap, where required
%            updates=neighborhoods(ismembc(neighborhoods,parts{dim})).';
%            sig_upd=locsigs(ismembc(neighborhoods,updates));
            updates=neighborhoods';
            sig_upd=locsigs;

%             for pt=1:length(updates)
%             if (any(removed==updates(pt)))
%                disp('uh oh');
%                disp(['Index already removed ' num2str(updates(pt))]);
%             end
%             end

            
%             heap_size = heap.size();
%             
%             heap.erase(sigs(updates),updates);
%             
%             if (heap.size() ~= (heap_size - length(updates)))
%                disp(['...and there it is: ' num2str(heap_size) ' - ' num2str(length(updates)) ' vs. ' num2str(heap.size())]);
%                disp(sigs(updates));
%                disp(updates);
%                error('oops');
%             end
%             
%             heap.insert(sig_upd,updates);

            heap.update(sig_upd,updates);
            sigs(neighborhoods)=locsigs;
         end
         
         current = current + 1;
         
         % Update the time bar.
         if (toc > 1)
            tic;
            h = timebar(x, n, msg, tstart, h);
         end
      end
      
      % Close the time bar, if still open.
      if (all(ishghandle(h, 'figure')))
         close(h);
      end
      
      disp(['...done in ' num2str(toc(tstart)) 's']);   
%   end

   disp(['Generated fallback neighborhood pool ' num2str(fellback) ' time(s).']);
   
%   x_max=max(cellfun(@length, removed_wgts));
%   y=nan(length(removed_wgts),x_max);
   
%   for dim=1:dimensions
%      y(dim,1:length(removed_wgts{dim}))=removed_wgts{dim};
%   end

   x_max=length(removed_wgts);
   y = removed_wgts;

   figure, plot(1:x_max,y);
   figure, plot(1:length(dual_size),dual_size);

%   results = points(:,~ismember(1:size(points,2),removed));
   results = removed;

end

%******************************************************************************
% parseInputs
%******************************************************************************
function [userParams] = parseInputs(varargin)
   % Support function to parse inputs into userParams structure
   %
   % Parameters:
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    userParams - .

   userParams = struct('radius', 0, 'neighbors', 0, 'counts', [], 'steps', []);
   
   if isempty(varargin)
      error('Damkjer:InvalidOptions', ...
            ['A neighborhood size must be specified, either directly or '...
             'via optional parameters.']);
   end
   
   if length(varargin) == 1 || ~isnumeric(varargin{2})
      value = varargin{1};
      
      if (isscalar(value)  && ...
          isreal(value)    && ...
          value >= 5)
         userParams.neighbors = fix(value);
      else
         error('Damkjer:InvalidCount', ...
               ['Number of Neighbors must be a real valued positive '...
                'integer greater or equal to 5: ' num2str(value)]);
      end
      
      varargin(1) = [];
   end
   
   % Parse the Property/Value pairs
   if rem(length(varargin), 2) ~= 0
      error('Damkjer:PropertyValueNotPair', ...
            ['Additional arguments must take the form of Property/Value '...
             'pairs']);
   end
   
   propertyNames = {'neighbors', 'radius', 'counts', 'steps'};

   while ~isempty(varargin)
      property = varargin{1};
      value    = varargin{2};
      
      % If the property has been supplied in a shortened form, lengthen it
      iProperty = find(strncmpi(property, propertyNames, length(property)));
      
      if isempty(iProperty)
         error('Damkjer:InvalidProperty', 'Invalid Property');
      elseif length(iProperty) > 1
         error('Damkjer:AmbiguousProperty', ...
               'Supplied shortened property name is ambiguous');
      end
      
      property = propertyNames{iProperty};
      
      switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
          userParams.neighbors = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Number of Neighbors must be a real valued positive '...
                   'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidRadius', ...
                  'Radius must be a real valued positive scalar');
         end
      case 'counts'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value >= 5))
            userParams.counts = fix(value);
         else
            error('Damkjer:InvalidCount', ...
                  ['Counts must be a sorted vector of real valued positive '...
                   'integers greater or equal to 5']);
         end
      case 'steps'
         if (isvector(value)  && ...
             isreal(value)    && ...
             issorted(value)  && ...
             all(value > 0))
            userParams.steps = value;
         else
            error('Damkjer:InvalidSteps', ...
                  ['Steps must be a sorted vector of real valued positive '...
                   'values']);
         end
      end
      
      varargin(1:2) = [];
   end

   % Check for mutually exclusive options.
   if (~isempty(userParams.counts) && userParams.neighbors >= 5)
      error('Damkjer:MutExOpts', ...
            '''neighbors'' and ''counts'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.steps) && userParams.radius > 0)
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''radius'' options are mutually exclusive');
   end
   
   if (~isempty(userParams.counts) && ~isempty(userParams.steps))
      error('Damkjer:MutExOpts', ...
            '''steps'' and ''counts'' options are mutually exclusive');
   end

   % Default, if necessary.
   if (userParams.neighbors <= 0 && ...
       userParams.radius <= 0 && ...
       isempty(userParams.counts) && ...
       isempty(userParams.steps))
      userParams.radius = 1;
   end
end
