% Sig_Dyn   Point Significance (Dyn et al)
%
% File: sig_Dyn.m
%
% Description:
%    Compute point significance according to Dyn 2004.
%
% Limitations:
%    No known limitations.
%
% Synopsis:
%    [results] = sig_Dyn(points, test_sets, neighborhood_sets)
%
% Inputs:
%    points            - .
%    test_sets         - .
%    neighborhood_sets - .
%
%
% Outputs:
%    results - Thinned point cloud.
%
% Toolbox requirements:
%    None.
%
% Code requirements:
%    None.
%
% Data requirements:
%    None.
%
% References:
%    Dyn reference.
%
% See Also:
%    None.
%

% Copyright (C)  2013 Kristian L. Damkjer.
%
%   Software History:
%      2013-DEC-28  Initial coding.
%

%******************************************************************************
% sig_Dyn
%******************************************************************************
function [ sigs ] = sig_Dyn( points, tests, nbrs )
   % Perform local feature attribution via eigenspace analysis.
   %
   % Parameters:
   %    points  - .
   %
   %    threshold - .
   %
   %    varargin - Variable-length input argument list. Option strings may be
   %               provided as abbreviations as long as they resolve to a
   %               unique option.Defined key-value pairs include the following:
   %               'neighbors' - .
   %               'radius'    - .
   %               'counts'    - .
   %               'steps'     - .
   %
   % Returns:
   %    classes - Structure containing the spatial analysis classifications.
   %

   % Sizing metrics
   dimensions=size(points, 1);
   elements=length(tests);

   % Pre-allocate classes
   norms=zeros(dimensions, elements);

   
   % Fit planes to the neighborhoods
   cells=cell(1,elements);
         
   for elem=1:elements
      cells{elem}=points(:,nbrs{elem}).'; % fastcov is expecting row-vectors
   end
         
   [covs,~,~]=fastcov(cells);

   % Compute the neighborhood covariance eigenvalues.
   [V,D]=par_eig(covs);

   sigs = zeros(1,elements);
   
   for elem=1:elements

      % skip underconstrained neighborhoods (possible with radius searches)
      if (length(nbrs{elem})<5 || any(any(isnan(covs{elem}))))
         continue;
      end

      [~,index]=min(abs(D{elem}));
      A=V{elem};
      aNorm=A(:,index);

      % Plane fit to the neighborhood is defined by the normal and the point.
      norms(:,elem)=aNorm;
      ys = points(:,tests{elem});
      x  = ys(:,1).';
            
      % Project neighborhood points to the fitted plane to establish local
      % coordinate reference frame.
      %
      % NOTE: null-space method only works when treating norm as row-vector.
      basis  = null(aNorm.');
      coeffs = bsxfun(@minus, cells{elem}, x) * basis;
      Z = bsxfun(@plus,  coeffs * basis.', x);

      % Compute RBF function values for the neighborhood observations.
      f=sqrt(sum(abs(cells{elem}-Z).^2,2));

      % Compute RBF coefficients
      
      % Need R's established as |Zi-Zj|
      % rM := radii Matrix
      rM = min(1,pdist2(Z,Z)./1);
      
      % PhiM := A_{Phi,Z} = [Phi(z_{i},z_{j})]
      %
      % The equation used below is a Wendland compactly supported basis
      % function in C2 to ensure curvature continuity. A C0 equivalent would be
      % (1-r)^2 which would only ensure positional continuity.
      PhiM = (1-rM).^4.*(4*rM+1);
      
%       if rcond(PhiM) < 1e-12
% %         fmt=get(0, 'format');
% %         format('hex');
%          disp(Z);
%          disp(rM);
%          disp(f);
%          disp(PhiM);
%          1-rM
%          (1-rM).^4
%          4*rM
%          4*rM+1
%          ((1-rM).^4).*(4*rM+1)
%          1-(((1-rM).^4).*(4*rM+1))
%          hold on
%          scatter3(cells{elem}(:,1),cells{elem}(:,2),cells{elem}(:,3), 5, 'blue', 'o');
%          scatter3(Z(:,1),Z(:,2),Z(:,3), 5, 'green', 'o');
%          scatter3(x(1),x(2),x(3), 5, 'red', 'x');
%          hold off
% %         set(0, 'format', fmt);
%          error('oops');
%       end
      
      as = PhiM\f;
      
      % radii from x.
      rawRx=sqrt(sum(abs(bsxfun(@minus, x, Z)).^2,2));
      rX = min(1,rawRx./1);
      PhiX = (1-rX).^4.*(4*rX+1);

      sigs(elem)=abs(as.' * PhiX);

%       if (sigs(elem) == 0)
%          hold on
%          scatter3(cells{elem}(:,1),cells{elem}(:,2),cells{elem}(:,3), 5, 'blue', 'o');
%          scatter3(Z(:,1),Z(:,2),Z(:,3), 5, 'green', 'o');
%          scatter3(x(1),x(2),x(3), 5, 'red', 'x');
%          hold off
%          
%          disp(rawRx);
%          error('oops');
%       end
   end

%    disp(sigs);
      
end
