//****************************************************************************
// FILE:        SplayTree.h
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   The interface definition for splay trees.
//<
//
// LIMITATIONS:
//>   This class template file follows the template inclusion pattern. This
//    header file should be the only file included by clients wishing to
//    instantiate a VpTree specialization.
//<
//
// SOFTWARE HISTORY:
//
//> 2013-DEC-02  K. Damkjer
//               Initial Coding.
//<
//****************************************************************************
#ifndef Damkjer_SplayTree_HEADER
#define Damkjer_SplayTree_HEADER

#include <deque>
#include <functional>

namespace Damkjer
{

//*****************************************************************************
// CLASS: SplayTree
//>   A splay tree data structure and related algorithms.
//
//    @tparam MetricT The metric search space.
//<
//*****************************************************************************
template<typename T, typename ComparitorT = std::less<T> >
class SplayTree
{
public:
   class Node
   {
   public:
      Node(const T& element = T())
         : theLBranch(0)
         , theRBranch(0)
         , theElement(element)
         , theIndex(0)
      {
      }
         //>
         //<

      ~Node()
      {
         std::deque<Node*> q;

         q.push_back(this);

         while (q.size() > 0)
         {
            Node* node = q.front();
            q.pop_front();

            if (node)
            {
               q.push_back(node->theLBranch);
               q.push_back(node->theRBranch);

               node->theLBranch = 0;
               node->theRBranch = 0;

               if (node != this) delete node;
            }
         }
      }
         //>
         //<

      Node* theLBranch;
      Node* theRBranch;
      T     theElement;
      std::size_t theIndex;
   };
      //> A node interface for a splay tree.
      //<

   SplayTree();
      //> Instantiate an empty tree.
      //<

   virtual ~SplayTree();
      //> Destruct the tree and deallocate resources.
      //<

   void insert(const T&);
      //> Add a new element to the tree.
      //<

   void update(const T&, const std::size_t&);
      //> Remove an element from the tree.
      //<
   
   void erase(const T&);
      //> Remove an element from the tree.
      //<

   void eraseIndex(const std::size_t&);
      //> Remove an element from the tree.
      //<

   Node* find(const T&);
      //> Find an element in the tree.
      //<

   const T& head();
      //> Find the front (minimum) element in the tree.
      //<

   const T& tail();
      //> Find the back (maximum) element in the tree.
      //<

   bool empty() const { return theRoot == 0; }
      //> Return true if and only if there are no elements in the tree.
      //<

   const std::size_t& size() const { return theSize; }
      //> Return the number of elements in the tree.
      //<

private:

   void splay(const T&);
      //> Perform the splay operation.
      //<

   Node* headFrom(Node*);
      //> Calculate the head (minimum) of the subtree rooted at the given node.
      //<

   Node* tailFrom(Node*);
      //> Calculate the tail (maximum) of the subtree rooted at the given node.
      //<

   ComparitorT theOrderIs;
      //> Pairwise comparitor that returns true if and only if the argument
      //  order of elements is in-order according to the comparitor.
      //<

   Node* theRoot;
      //> The root is the starting point for all operations against the database
      //  represented by the SplayTree.
      //<

   std::size_t theSize;
      //> The total number of elements contained in the tree.
      //<
   
   std::deque<Node*> theIndexedList;
};

}

//*****************************************************************************
// External References
//*****************************************************************************
//#include "SplayTree_Node.h"

//*****************************************************************************
// Template Inclusion
//*****************************************************************************
#include "SplayTree.hpp" // HASA SplayTree Template Implementation.

#endif
