//****************************************************************************
// FILE:        SplayTree.hpp
//
//    Copyright (C)  2013 Kristian Damkjer.
//
// DESCRIPTION:
//>   The template implementation for splay trees.
//<
//
// LIMITATIONS:
//>   This class template file follows the template inclusion pattern. This
//    header file should be the only file included by clients wishing to
//    instantiate a VpTree specialization.
//<
//
// SOFTWARE HISTORY:
//
//> 2013-DEC-02  K. Damkjer
//               Initial Coding.
//<
//****************************************************************************
#include "SplayTree.h"

namespace Damkjer
{

//*****************************************************************************
// SplayTree::SplayTree()
//>   Instantiate an empty tree.
//
//    @tparam T            The element type.
//    @tparam ComparitorT  The element comparitor type.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
SplayTree<T, ComparitorT>::SplayTree()
   : theRoot(0)
   , theSize(0)
   , theIndexedList(1)
{
}

//*****************************************************************************
// SplayTree::~SplayTree()
//>   Destruct the tree and deallocate resources.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
SplayTree<T, ComparitorT>::~SplayTree()
{
   delete theRoot;
   theRoot = 0;
}

//*****************************************************************************
// SplayTree::insert(const T&)
//>   Add a new element to the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
void
SplayTree<T, ComparitorT>::insert(const T& element)
{
   Node* node = new Node(element);
   node->theIndex = theIndexedList.size();
   theIndexedList.push_back(node);

   if (!theRoot)
   {
      theRoot = node;
      ++theSize;
      return;
   }

   splay(element);

   if (theOrderIs(element, theRoot->theElement))
   {
      node->theLBranch = theRoot->theLBranch;
      node->theRBranch = theRoot;
      theRoot->theLBranch = 0;
      theRoot = node;
      ++theSize;
   }
   else // if (theOrderIs(theRoot->theElement, element))
   {
      node->theRBranch = theRoot->theRBranch;
      node->theLBranch = theRoot;
      theRoot->theRBranch = 0;
      theRoot = node;
      ++theSize;
   }
//   else
//   {
//      delete node;
//   }

#if USE_OBSOLETE
   Node* current = theRoot;
   Node* parent  = 0;

   while (current)
   {
      parent = current;

      if (theOrderIs(current->theElement, element))
      {
         current = current->theRBranch;
      }
      else
      {
         current = current->theLBranch;
      }
   }

   current = new Node(element);
   current->theParent = parent;

   if (!parent)
   {
      theRoot = current;
   }
   else if (theOrderIs(parent->theElement, element))
   {
      parent->theRBranch = current;
   }
   else
   {
      parent->theLBranch = current;
   }

   splay(current);

   ++theSize;
#endif
}

//*****************************************************************************
// SplayTree::update(const T&)
//>   Add a new element to the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
void
SplayTree<T, ComparitorT>::update(const T& element, const std::size_t& index)
{
   // This can probably be made more efficient, but this will suffice for now.
   Node* found = theIndexedList[index];

   if (!found) return; // What? Who's mismanaging the structure?

   eraseIndex(index);

   Node* node = new Node(element);
   node->theIndex = index;
   theIndexedList[index] = node;

   //if (!theRoot) // How can we be found, but not have a root?
   //{
   //   theRoot = node;
   //   ++theSize;
   //   return;
   //}

   splay(element);

   if (theOrderIs(element, theRoot->theElement))
   {
      node->theLBranch = theRoot->theLBranch;
      node->theRBranch = theRoot;
      theRoot->theLBranch = 0;
      theRoot = node;
      ++theSize;
   }
   else // if (theOrderIs(theRoot->theElement, element))
   {
      node->theRBranch = theRoot->theRBranch;
      node->theLBranch = theRoot;
      theRoot->theRBranch = 0;
      theRoot = node;
      ++theSize;
   }
}

//*****************************************************************************
// SplayTree::erase(const T&)
//>   Remove an element from the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
void
SplayTree<T, ComparitorT>::erase(const T& element)
{
   Node* found = find(element);

   if (!found) return;

   if (!found->theLBranch)
   {
      theRoot = found->theRBranch;
   }
   else
   {
      theRoot = found->theLBranch;
      splay(element);
      theRoot->theRBranch = found->theRBranch;
   }

   --theSize;
   
   found->theLBranch = 0;
   found->theRBranch = 0;
   theIndexedList[found->theIndex] = 0;
   delete found;

#if USE_OBSOLETE
   Node* found = find(element);

   if (!found) return;

   // No need to splay here... splay performed by find.

   if (!found->theLBranch)
   {
      replace(found, found->theRBranch);
   }
   else if (!found->theRBranch)
   {
      replace(found, found->theLBranch);
   }
   else
   {
      Node* newRoot = headFrom(found->theRBranch);

      // NOTE: by definition, newRoot has no left child

      if (newRoot->theParent != found)
      {
         replace(newRoot, newRoot->theRBranch);
         newRoot->theRBranch = found->theRBranch;
         newRoot->theRBranch->theParent = newRoot;
      }

      replace(found, newRoot);
      newRoot->theLBranch = found->theLBranch;
      newRoot->theLBranch->theParent = newRoot;
   }

   found->theLBranch = 0;
   found->theRBranch = 0;
   found->theParent = 0;
   delete found;
   --theSize;
#endif
}

//*****************************************************************************
// SplayTree::erase_index(const std::size_t&)
//>   Remove an element from the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
void
SplayTree<T, ComparitorT>::eraseIndex(const std::size_t& index)
{
   Node* found = theIndexedList.at(index);

   if (!found) return;

   splay(found->theElement);
   
   if (!found->theLBranch)
   {
      theRoot = found->theRBranch;
   }
   else
   {
      theRoot = found->theLBranch;
      splay(found->theElement);
      theRoot->theRBranch = found->theRBranch;
   }

   --theSize;
   
   found->theLBranch = 0;
   found->theRBranch = 0;
   theIndexedList[index] = 0;
   delete found;
}

//*****************************************************************************
// SplayTree::find(const T&)
//>   Find an element in the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
typename SplayTree<T, ComparitorT>::Node*
SplayTree<T, ComparitorT>::find(const T& element)
{
   splay(element);

   if (theOrderIs(theRoot->theElement, element))
   {
      return 0;
   }

   if (theOrderIs(element, theRoot->theElement))
   {
      return 0;
   }

   return theRoot;

#if USE_OBSOLETE
   Node* previous = 0;
   Node* current  = theRoot;

   while (current)
   {
      if (theOrderIs(current->theElement, element))
      {
         previous = current;
         current = current->theRBranch;
      }
      else if (theOrderIs(element, current->theElement))
      {
         previous = current;
         current = current->theLBranch;
      }
      else
      {
         splay(current);
         return current;
      }
   }

   if (previous) splay(previous);

   return 0;
#endif
}

//*****************************************************************************
// SplayTree::head()
//>   Find the minimum element in the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
const T&
SplayTree<T, ComparitorT>::head()
{
   return headFrom(theRoot)->theElement;
}

//*****************************************************************************
// SplayTree::tail()
//>   Find the maximum element in the tree.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
const T&
SplayTree<T, ComparitorT>::tail()
{
   return tailFrom(theRoot)->theElement;
}

//*****************************************************************************
// SplayTree::splay(Node* const)
//>   Perform the splay operation.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
void
SplayTree<T, ComparitorT>::splay(const T& element)
{
   if (!theRoot) return;

   Node* node = theRoot;
   Node* child = 0;

   Node temp;
   Node* l = &temp;
   Node* r = &temp;

   for(;;)
   {
      if (theOrderIs(element, node->theElement))
      {
         if (!node->theLBranch) break;

         if (theOrderIs(element, node->theLBranch->theElement))
         {
            child = node->theLBranch;
            node->theLBranch = child->theRBranch;
            child->theRBranch = node;
            node = child;

            if (!node->theLBranch) break;
         }

         r->theLBranch = node;
         r = node;
         node = node->theLBranch;
      }
      else if (theOrderIs(node->theElement, element))
      {
         if (!node->theRBranch) break;

         if (theOrderIs(node->theRBranch->theElement, element))
         {
            child = node->theRBranch;
            node->theRBranch = child->theLBranch;
            child->theLBranch = node;
            node = child;
            if (!node->theRBranch) break;
         }

         l->theRBranch = node;
         l = node;
         node = node->theRBranch;
      }
      else
      {
         break;
      }
   }

   l->theRBranch = node->theLBranch;
   r->theLBranch = node->theRBranch;
   node->theLBranch = temp.theRBranch;
   node->theRBranch = temp.theLBranch;
   temp.theRBranch = 0;
   temp.theLBranch = 0;

   theRoot = node;

#if USE_OBSOLETE
   while (bubble->theParent)
   {
      if (bubble->theParent->theLBranch == bubble)
      {
         // CASE: The bubble is a left-child

         if (!bubble->theParent->theParent)
         {
            // CASE: Zig Condition.
            zig(bubble->theParent);
         }
         else if (bubble->theParent->theParent->theLBranch == bubble->theParent)
         {
            // CASE: Zig-Zig Condition
            zig(bubble->theParent->theParent);
            zig(bubble->theParent);
         }
         else
         {
            // CASE: Zig-Zag Condition
            zig(bubble->theParent);
            zag(bubble->theParent);
         }
      }
      else
      {
         // CASE: The bubble is a right-child

         if (!bubble->theParent->theParent)
         {
            // CASE: Zag Condition.
            zag(bubble->theParent);
         }
         else if (bubble->theParent->theParent->theRBranch == bubble->theParent)
         {
            // Zag-Zag Condition
            zag(bubble->theParent->theParent);
            zag(bubble->theParent);
         }
         else
         {
            // Zag-Zig Condition
            zag(bubble->theParent);
            zig(bubble->theParent);
         }
      }
   }
#endif
}
   
//*****************************************************************************
// SplayTree::headFrom(Node*)
//>   Calculate the head (minimum) of the subtree rooted at the given node.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
typename SplayTree<T, ComparitorT>::Node*
SplayTree<T, ComparitorT>::headFrom(Node* current)
{
   if (!current) return 0;
   
   while (current->theLBranch) current = current->theLBranch;

   return current;
}

//*****************************************************************************
// SplayTree::tailFrom(Node*)
//>   Calculate the tail (maximum) of the subtree rooted at the given node.
//<
//*****************************************************************************
template<typename T, typename ComparitorT>
typename SplayTree<T, ComparitorT>::Node*
SplayTree<T, ComparitorT>::tailFrom(Node* current)
{
   if (!current) return 0;
   
   while (current->theRBranch) current = current->theRBranch;

   return current;
}

}
