% SplayTree   Splay Tree indexed search data structure
%
% File: SearchTree.m
%
% Description:
%    This MATLAB class represents a splay tree data structure. The
%    class is derived from the MATLAB handle class to provide automated object
%    descruction on calls to "clear".
%
% Limitations:
%    See SplayTree.h for complete description of limitations.
%
% Properties:
%    None.
%
% Methods:
%        [tree] = SplayTree(varargin)              - VP Tree Constructor
%                 delete(tree)                     - Release this object and
%                                                    free memory.
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    None.
%
% Data requirements:
%    SplayTreeAPI MEX library.
%
% References:
%    Put the Splay Tree paper reference here. If the line wraps (which it
%       probably will), then indent the subsequent lines.
%
% See Also:
%    None.
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2013-DEC-28   K. Damkjer
%       Initial Coding.
%

%******************************************************************************
% CLASS: SplayTree
%    Splay Tree indexed search data structure. The class is derived
%    from the MATLAB handle class to provide automated object descruction on
%    calls to "clear".
%******************************************************************************
classdef (Sealed = true) SplayTree < handle
  
   properties (Hidden=true, Access = private)
      % The C++ Splay-tree object owned by this handle class.
      theTree
   end
   
   methods
      %************************************************************************
      % SplayTree/SplayTree
      %************************************************************************
      function this = SplayTree(weights, elems)
         % Build the index structure for the database.
         %
         % Parameters:
         %    varargin - Variable-length input argument list. Currently, only a
         %               single argument is permitted in the argument list.
         %               Specifically, a set of point observations that will be
         %               spatially indexed by the tree structure.
         
         if (nargin ~= 2)
            error('SplayTree:InvalidArgumentList',...
                  'SplayTree must be constructed with a set of indexes');
         end

         this.theTree = SplayTreeAPI('create', weights, uint64(elems));
      end

      %************************************************************************
      % SplayTree/delete
      %************************************************************************
      function delete(this)
         % This function frees memory on delete (clear) of the SplayTree.
         SplayTreeAPI('destroy', this.theTree);
      end
      
      %************************************************************************
      % VpTree/pop_head
      %************************************************************************
      function [wgt,idx] = pop_head(this)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         [wgt,idx] = SplayTreeAPI('pop_head', this.theTree);
      end

      %************************************************************************
      % VpTree/head
      %************************************************************************
      function [wgt,idx] = head(this)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         [wgt,idx] = SplayTreeAPI('head', this.theTree);
      end
      
      %************************************************************************
      % VpTree/insert
      %************************************************************************
      function insert(this, weights, elems)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         SplayTreeAPI('insert', this.theTree, weights, elems);
      end
      
      %************************************************************************
      % VpTree/update
      %************************************************************************
      function update(this, weights, elems)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         SplayTreeAPI('update', this.theTree, weights, elems);
      end
      
      %************************************************************************
      % VpTree/erase
      %************************************************************************
      function erase(this, weights, elems)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         SplayTreeAPI('erase', this.theTree, weights, elems);
      end
      
      %************************************************************************
      % VpTree/erase_index
      %************************************************************************
      function erase_index(this, elems)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         SplayTreeAPI('erase_index', this.theTree, elems);
      end
      
      %************************************************************************
      % VpTree/size
      %************************************************************************
      function [tree_size] = size(this)
         % Perform a k-nearest neighbor search on the database with the
         % set of queries.
         %
         % Parameters:
         %
         % Returns:
         %
         
         [tree_size] = SplayTreeAPI('size', this.theTree);
      end
   end
end

