% geoKeyFactory   Concrete factory for GeoTIFF key objects
%
% File:
%    geoKeyFactory.m
%
% Description:
%    This MATLAB function provides an interface similar to a concrete factory
%    in the abstract factory design pattern for constructing defined GeoTIFF key
%    objects.
%
% Limitations:
%    Several of the concrete classes referenced by this factory do not exist.
%    This means that an error will be thrown if they key is recognized, but no
%    corresponding class exists. This was done to allow me to incrementally
%    develop classes for keys as they are encountered.
%
% Synopsis:
%    [key] = geoKeyFactory(code, location, size, data)
%
% Inputs:
%    vlrHeader  - A LAS VLR object containing populated header data
%
% Outputs:
%    vlr  - A specialized LAS File VLR Specification object
%
% Other m-files required:
%    GTModelTypeGeoKey.m
%    GTRasterTypeGeoKey.m
%    GeographicTypeGeoKey.m
%    GeogGeodeticDatumGeoKey.m
%    GeogPrimeMeridianGeoKey.m
%    GeogLinearUnitsGeoKey.m
%    GeogAngularUnitsGeoKey.m
%    GeogEllipsoidGeoKey.m
%    GeogAzimuthUnitsGeoKey.m
%    ProjectedCSTypeGeoKey.m
%    ProjectionGeoKey.m
%    ProjCoordTransGeoKey.m
%    ProjLinearUnitsGeoKey.m
%    VerticalCSTypeGeoKey.m
%    VerticalDatumGeoKey.m
%    VerticalUnitsGeoKey.m
%    GeogLinearUnitSizeGeoKey.m
%    GeogAngularUnitSizeGeoKey.m
%    GeogSemiMajorAxisGeoKey.m
%    GeogSemiMinorAxisGeoKey.m
%    GeogInvFlatteningGeoKey.m
%    GeogPrimeMeridianLongGeoKey.m
%    ProjLinearUnitSizeGeoKey.m
%    ProjStdParallel1GeoKey.m
%    ProjStdParallel2GeoKey.m
%    ProjNatOriginLongGeoKey.m
%    ProjNatOriginLatGeoKey.m
%    ProjFalseEastingGeoKey.m
%    ProjFalseNorthingGeoKey.m
%    ProjFalseOriginLongGeoKey.m
%    ProjFalseOriginLatGeoKey.m
%    ProjFalseOriginEastingGeoKey.m
%    ProjFalseOriginNorthingGeoKey.m
%    ProjCenterLongGeoKey.m
%    ProjCenterLatGeoKey.m
%    ProjCenterEastingGeoKey.m
%    ProjCenterNorthingGeoKey.m
%    ProjScaleAtNatOriginGeoKey.m
%    ProjScaleAtCenterGeoKey.m
%    ProjAzimuthAngleGeoKey.m
%    ProjStraightVertPoleLongGeoKey.m
%    GTCitationGeoKey.m
%    GeogCitationGeoKey.m
%    PCSCitationGeoKey.m
%    VerticalCitationGeoKey.m
%
% MAT-files required:
%    None.
%
% References:
%    http://www.remotesensing.org/geotiff/spec/geotiffhome.html
%
% See Also:
%    GeoKey
%    GTModelTypeGeoKey
%    GTRasterTypeGeoKey
%    GeographicTypeGeoKey
%    GeogGeodeticDatumGeoKey
%    GeogPrimeMeridianGeoKey
%    GeogLinearUnitsGeoKey
%    GeogAngularUnitsGeoKey
%    GeogEllipsoidGeoKey
%    GeogAzimuthUnitsGeoKey
%    ProjectedCSTypeGeoKey
%    ProjectionGeoKey
%    ProjCoordTransGeoKey
%    ProjLinearUnitsGeoKey
%    VerticalCSTypeGeoKey
%    VerticalDatumGeoKey
%    VerticalUnitsGeoKey
%    GeogLinearUnitSizeGeoKey
%    GeogAngularUnitSizeGeoKey
%    GeogSemiMajorAxisGeoKey
%    GeogSemiMinorAxisGeoKey
%    GeogInvFlatteningGeoKey
%    GeogPrimeMeridianLongGeoKey
%    ProjLinearUnitSizeGeoKey
%    ProjStdParallel1GeoKey
%    ProjStdParallel2GeoKey
%    ProjNatOriginLongGeoKey
%    ProjNatOriginLatGeoKey
%    ProjFalseEastingGeoKey
%    ProjFalseNorthingGeoKey
%    ProjFalseOriginLongGeoKey
%    ProjFalseOriginLatGeoKey
%    ProjFalseOriginEastingGeoKey
%    ProjFalseOriginNorthingGeoKey
%    ProjCenterLongGeoKey
%    ProjCenterLatGeoKey
%    ProjCenterEastingGeoKey
%    ProjCenterNorthingGeoKey
%    ProjScaleAtNatOriginGeoKey
%    ProjScaleAtCenterGeoKey
%    ProjAzimuthAngleGeoKey
%    ProjStraightVertPoleLongGeoKey
%    GTCitationGeoKey
%    GeogCitationGeoKey
%    PCSCitationGeoKey
%    VerticalCitationGeoKey
%

% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

function [key] = geoKeyFactory(code, location, size, data)
switch location
    case 0
        switch code
            case 1024
                key = GTModelTypeGeoKey(data);
            case 1025
                key = GTRasterTypeGeoKey(data);
            case 2048
                key = GeographicTypeGeoKey(data);
            case 2050
                key = GeogGeodeticDatumGeoKey(data);
            case 2051
                key = GeogPrimeMeridianGeoKey(data);
            case 2052
                key = GeogLinearUnitsGeoKey(data);
            case 2054
                key = GeogAngularUnitsGeoKey(data);
            case 2056
                key = GeogEllipsoidGeoKey(data);
            case 2060
                key = GeogAzimuthUnitsGeoKey(data);
            case 3072
                key = ProjectedCSTypeGeoKey(data);
            case 3074
                key = ProjectionGeoKey(data);
            case 3075
                key = ProjCoordTransGeoKey(data);
            case 3076
                key = ProjLinearUnitsGeoKey(data);
            case 4096
                key = VerticalCSTypeGeoKey(data);
            case 4098
                key = VerticalDatumGeoKey(data);
            case 4099
                key = VerticalUnitsGeoKey(data);
            otherwise
                error('geoKeyFactory:InvalidGeoNativeCode',...
                      'Unrecognized Code for Native GeoKey');
        end
    case 34736
        switch code
            case 2053
                key = GeogLinearUnitSizeGeoKey(data);
            case 2055
                key = GeogAngularUnitSizeGeoKey(data);
            case 2057
                key = GeogSemiMajorAxisGeoKey(data);
            case 2058
                key = GeogSemiMinorAxisGeoKey(data);
            case 2059
                key = GeogInvFlatteningGeoKey(data);
            case 2061
                key = GeogPrimeMeridianLongGeoKey(data);
            case 3077
                key = ProjLinearUnitSizeGeoKey(data);
            case 3078
                key = ProjStdParallel1GeoKey(data);
            case 3079
                key = ProjStdParallel2GeoKey(data);
            case 3080
                key = ProjNatOriginLongGeoKey(data);
            case 3081
                key = ProjNatOriginLatGeoKey(data);
            case 3082
                key = ProjFalseEastingGeoKey(data);
            case 3083
                key = ProjFalseNorthingGeoKey(data);
            case 3084
                key = ProjFalseOriginLongGeoKey(data);
            case 3085
                key = ProjFalseOriginLatGeoKey(data);
            case 3086
                key = ProjFalseOriginEastingGeoKey(data);
            case 3087
                key = ProjFalseOriginNorthingGeoKey(data);
            case 3088
                key = ProjCenterLongGeoKey(data);
            case 3089
                key = ProjCenterLatGeoKey(data);
            case 3090
                key = ProjCenterEastingGeoKey(data);
            case 3091
                key = ProjCenterNorthingGeoKey(data);
            case 3092
                key = ProjScaleAtNatOriginGeoKey(data);
            case 3093
                key = ProjScaleAtCenterGeoKey(data);
            case 3094
                key = ProjAzimuthAngleGeoKey(data);
            case 3095
                key = ProjStraightVertPoleLongGeoKey(data);
            otherwise
               error('geoKeyFactory:InvalidGeoDoubleCode',...
                     'Unrecognized Code for Double GeoKey');
        end
    case 34767
        switch code
            case 1026
                key = GTCitationGeoKey(data,size);
            case 2049
                key = GeogCitationGeoKey(data,size);
            case 3073
                key = PCSCitationGeoKey(data,size);
            case 4097
                key = VerticalCitationGeoKey(data,size);
            otherwise
               error('geoKeyFactory:InvalidGeoAsciiCode',...
                     'Unrecognized Code for Ascii GeoKey');
        end
    otherwise
       error('geoKeyFactory:InvalidLocation',...
             'Unrecognized Data Location for GeoKey');
end
end
