% LasFile   ASPRS LAS format data structure
%
% File:
%    LasFile.m
%
% Description:
%    This MATLAB class represents an ASPRS LAS version 1.2 file.
%
% Limitations:
%    There are currently no mechanisms for setting VLR data. This means that it
%    is not possible to geocode LAS files with this class.
%
% Properties:
%    publicHeader      - The LAS File Public Header Block
%    vlrs              - The LAS File VLRs
%    intensity         - The Pulse Return Magnitude
%    returnNumber      - The Pulse Return Number
%    numberOfReturns   - The Pulse Number of Returns
%    scanDirection     - The Pulse Scan Direction
%    edgeOfFlightLine  - Edge of Flight Line Flag
%    classification    - The Pulse Category Classification
%    synthetic         - The Synthetic Point Flag
%    keyPoint          - The Key Point Flag
%    withheld          - The Withheld Point Flag
%    scanAngleRank     - The Pulse Scan Angle Rank
%    userData          - The User Data Field
%    pointSourceID     - The Point Source ID
%    points            - The Pulse X, Y, and Z Coordinates
%    gpsTime           - The Pulse GPS Time
%    color             - The Point Color
%
% Methods:
%    [lasFile] = LasFile(varargin)   - Constructor for LasFile objects.
%    [lasFile] = loadFrom(location)  - Load LAS file from a given location. 
%                saveTo(location)    - Save LAS file to a given location.
%
% Toolbox requirements:
%    None.
%
% Script requirements:
%    LasPublicHeader.m
%    LasVlr.m
%    lasVlrFactory.m
%
% Data requirements:
%    None.
%
% References:
%    http://asprs.org/a/society/committees/standards/asprs_las_format_v12.pdf
%
% See Also:
%    LasPublicHeader
%    LasVlr
%    lasVlrFactory
%

% Copyright (C)  2012 Kristian L. Damkjer.
%
% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

classdef (Sealed = true) LasFile
   properties
      % The LAS File Public Header Block
      %
      % The public header block contains information about the LAS file
      % that is used to document the details of this file for parsing and
      % writing.
      publicHeader = LasPublicHeader
      
      % The LAS File VLRs
      %
      % The Public Header Block is followed by one or more Variable
      % Length Records. Each VLR header is 54 bytes in length with the
      % data section dependent on the specific VLR.
      vlrs
      
      % The Pulse Return Magnitude
      %
      % The intensity value is the integer representation of the pulse
      % return magnitude. This value is optional and system specific.
      % However, it should always be included if available.
      intensity
      
      % The Pulse Return Number
      %
      % The Return Number is the pulse return number for a given output
      % pulse. A given output laser pulse can have many returns, and they
      % must be marked in sequence of return. The first return will have
      % a Return Number of one, the second a Return Number of two, and so
      % on up to five returns.
      returnNumber
      
      % The Pulse Number of Returns
      %
      % The Number of Returns is the total number of returns for a given
      % pulse. For example, a laser data point may be return two (Return
      % Number) within a total of five returns.
      numberOfReturns
      
      % The Pulse Scan Direction
      %
      % The Scan Direction Flag denotes the direction at which the
      % scanner mirror was travelling at the time of the output pulse. A
      % bit value of 1 is a positive scan direction, and a bit value of 0
      % is a negative scan direction (where positive scan direction is a
      % scan moving from the left side of the in-track direction to the
      % right side and negative the opposite).
      scanDirection
      
      % Edge of Flight Line Flag
      %
      % The Edge of Flight Line data bit has a value of 1 only when the
      % point is at the end of a scan. It is the last point on a given
      % scan line before it changes direction.
      edgeOfFlightLine
      
      % The Pulse Category Classification
      %
      % Classification in LAS 1.0 was essentially user defined and
      % optional. Las 1.1 defines a standard set of ASPRS
      % classifications. In addition, the field is now mandatory. If a
      % point has never been classified, this byte must be set to zero.
      % There are no user defined classes since both point format 0 and
      % point format 1 supply 8 bits per point for user defined
      % operations.
      %
      % Value | Meaning
      % ------+----------------------------------------------------------
      %  0    | Created, never classified
      %  1    | Unclassified
      %  2    | Ground
      %  3    | Low Vegetation
      %  4    | Medium Vegetation
      %  5    | High Vegetation
      %  6    | Building
      %  7    | Low Point (noise)
      %  8    | Model Key-point (mass point)
      %  9    | Water
      % 10    | Reserved for ASPRS Definition
      % 11    | Reserved for ASPRS Definition
      % 12    | Overlap Point
      % 13-31 | Reserved for ASPRS Definition
      classification
      
      % The Synthetic Point Flag
      %
      % If set then this point was created by a technique other than
      % LIDAR collection such as digitized from a photogrammetric stereo
      % model.
      synthetic
      
      % The Key Point Flag
      %
      % If set then this point is considered to be a model key-point and
      % thus should generally not be withheld in a thinning algorithm.
      keyPoint
      
      % The Withheld Point Flag
      %
      % If set then this point should not be included in processing
      % (synonymous with Deleted).
      withheld
      
      % The Pulse Scan Angle Rank
      %
      % The Scan Angle Rank is a signed one-byte number with a valid
      % range from -90 to +90. The scan angle Rank is the angle (rounded
      % to the nearest integer in the absolute value sense) at which the
      % laser point was output from the laser system including the roll
      % of the aircraft. The scan angle is within 1 degree of accuracy
      % from +90 to -90 degrees. The scan angle is an angle based on 0
      % degrees being nadir, and -90 degrees to the left side of the
      % aircraft in the direction of flight.
      scanAngleRank
      
      % The User Data Field
      %
      % This field may be used at the user's discretion.
      userData
      
      % The Point Source ID
      %
      % This value indicates the file from which this point originated.
      % Valid values for this field are 1 to 65,535 inclusive with zero
      % being used for a special case discussed below. The numerical
      % value corresponds to the File Source ID from which this point
      % originated. Zero is reserved as a convenience to system
      % implementers. A Point Source ID of zero implies that this point
      % originated in this file. This implies that processing software
      % should set the Point Source ID equal to the File Source ID of the
      % file containing this point at some time during processing.
      pointSourceID
   end
   
   properties (Access = private)
      % The Pulse X, Y, and Z Coordinates
      %
      % The X, Y, and Z values are stored as long inegers. The X, Y, and
      % Z values are used in conjunction with the scale values and the
      % offset values to determine the coordinate for each point as
      % described in the Public Header Block section.
      pPoints
      
      % The Pulse GPS Time
      %
      % The GPS Time is the double floating point time tag value at which
      % the point was acquired. It is GPS Week Time if the Global
      % Encoding low bit is clear and POSIX Time if the Global Encoding
      % low bit is set (see Global Encoding in the Public Header Block
      % description).
      pGpsTime
      
      % The Point Color
      %
      % Red  : The Red image channel value associated with this point
      % Green: The Green image channel value associated with this point
      % Blue : The Blue imagechannel value associated with this point
      pColor
   end
   
   properties (Dependent)
      % The Pulse X, Y, and Z Coordinates
      %
      % The X, Y, and Z values are stored as long integers. The X, Y, and
      % Z values are used in conjunction with the scale values and the
      % offset values to determine the coordinate for each point as
      % described in the Public Header Block section.
      points
      
      % The Pulse GPS Time
      %
      % The GPS Time is the double floating point time tag value at which
      % the point was acquired. It is GPS Week Time if the Global
      % Encoding low bit is clear and POSIX Time if the Global Encoding
      % low bit is set (see Global Encoding in the Public Header Block
      % description).
      gpsTime
      
      % The Point Color
      %
      % Red  : The Red image channel value associated with this point
      % Green: The Green image channel value associated with this point
      % Blue : The Blue imagechannel value associated with this point
      color
   end
   
   methods
      function lasFile = LasFile(varargin)
         % Constructor for LasFile objects
         %
         % LasFile objects may be constructed either by using a
         % no-argument (default) constructor or by passing a file
         % descriptor or file name.
         
         if (nargin == 0)
            lasFile.pPoints=zeros(3,0,'double');
            lasFile.intensity=zeros(1,0,'uint16');
            lasFile.returnNumber=zeros(1,0,'uint8');
            lasFile.numberOfReturns=zeros(1,0,'uint8');
            lasFile.scanDirection=false(1,0);
            lasFile.edgeOfFlightLine=false(1,0);
            lasFile.classification=zeros(1,0,'uint8');
            lasFile.synthetic=false(1,0);
            lasFile.keyPoint=false(1,0);
            lasFile.withheld=false(1,0);
            lasFile.scanAngleRank=zeros(1,0,'int8');
            lasFile.userData=zeros(1,0,'uint8');
            lasFile.pointSourceID=zeros(1,0,'uint16');
         elseif (nargin == 1)
            lasFile = lasFile.loadFrom(varargin{1});
         end
      end
      
      function pointData = get.points(lasFile)
         % Required accessor for dependent property: points
         pointData = lasFile.pPoints;
      end
      
      function timeData = get.gpsTime(lasFile)
         % Required accessor for dependent property: gpsTime
         timeData = lasFile.pGpsTime;
      end
      
      function colorData = get.color(lasFile)
         % Required accessor for dependent property: color
         colorData = lasFile.pColor;
      end
      
      function lasFile = set.points(lasFile, pointData)
         % Controlled interface for setting point data on LasFile objects
         if (size(pointData,1) ~= 3)
            error('LasFile:DimensionMismatch',...
               'Points must have 3 coordinates');
         end
         
         nPoints = size(pointData,2);
         
         if (size(lasFile.pPoints,2) > nPoints)
            lasFile.intensity=lasFile.intensity(:,1:nPoints);
            lasFile.returnNumber=lasFile.returnNumber(:,1:nPoints);
            lasFile.numberOfReturns=lasFile.numberOfReturns(:,1:nPoints);
            lasFile.scanDirection=lasFile.scanDirection(:,1:nPoints);
            lasFile.edgeOfFlightLine=lasFile.edgeOfFlightLine(:,1:nPoints);
            lasFile.classification=lasFile.classification(:,1:nPoints);
            lasFile.synthetic=lasFile.synthetic(:,1:nPoints);
            lasFile.keyPoint=lasFile.keyPoint(:,1:nPoints);
            lasFile.withheld=lasFile.withheld(:,1:nPoints);
            lasFile.scanAngleRank=lasFile.scanAngleRank(:,1:nPoints);
            lasFile.userData=lasFile.userData(:,1:nPoints);
            lasFile.pointSourceID=lasFile.pointSourceID(:,1:nPoints);
         else
            addPoints = nPoints-size(lasFile.pPoints,2);
            lasFile.intensity=[lasFile.intensity zeros(1,addPoints,'uint16')];
            lasFile.returnNumber=[lasFile.returnNumber zeros(1,addPoints,'uint8')];
            lasFile.numberOfReturns=[lasFile.numberOfReturns zeros(1,addPoints,'uint8')];
            lasFile.scanDirection=[lasFile.scanDirection false(1,addPoints)];
            lasFile.edgeOfFlightLine=[lasFile.edgeOfFlightLine false(1,addPoints)];
            lasFile.classification=[lasFile.classification zeros(1,addPoints,'uint8')];
            lasFile.synthetic=[lasFile.synthetic false(1,addPoints)];
            lasFile.keyPoint=[lasFile.keyPoint false(1,addPoints)];
            lasFile.withheld=[lasFile.withheld false(1,addPoints)];
            lasFile.scanAngleRank=[lasFile.scanAngleRank zeros(1,addPoints,'int8')];
            lasFile.userData=[lasFile.userData zeros(1,addPoints,'uint8')];
            lasFile.pointSourceID=[lasFile.pointSourceID zeros(1,addPoints,'uint16')];
         end
         
         lasFile.pPoints = pointData;
         
         lasFile.publicHeader.nPoints = size(pointData,2);
         lasFile.publicHeader.nPointsByReturn(1) = size(pointData,2);
         maxX = max(pointData(1,:));
         minX = min(pointData(1,:));
         maxY = max(pointData(2,:));
         minY = min(pointData(2,:));
         maxZ = max(pointData(3,:));
         minZ = min(pointData(3,:));
         lasFile.publicHeader.maxX = maxX;
         lasFile.publicHeader.minX = minX;
         lasFile.publicHeader.maxY = maxY;
         lasFile.publicHeader.minY = minY;
         lasFile.publicHeader.maxZ = maxZ;
         lasFile.publicHeader.minZ = minZ;
         lasFile.publicHeader.xOffset = minX+0.5*(maxX-minX);
         lasFile.publicHeader.yOffset = minY+0.5*(maxY-minY);
         lasFile.publicHeader.zOffset = minZ+0.5*(maxZ-minZ);
         lasFile.publicHeader.xScaleFactor = (maxX-minX)/...
            double(intmax('uint32')-1);
         lasFile.publicHeader.yScaleFactor = (maxY-minY)/...
            double(intmax('uint32')-1);
         lasFile.publicHeader.zScaleFactor = (maxZ-minZ)/...
            double(intmax('uint32')-1);
      end
      
      function lasFile = set.gpsTime(lasFile, timeData)
         % Controlled interface for setting GPS time data on LasFile objects
         
         if (size(timeData,2) ~= lasFile.publicHeader.nPoints)
            error('LasFile:DimensionMismatch',...
               'GPS Time Data must have an entry for each point');
         end
         
         lasFile.pGpsTime = timeData;
         
         lasFile.publicHeader.pointDataFormatID = ...
            bitor(lasFile.publicHeader.pointDataFormatID,1);
         
         if (lasFile.publicHeader.pointDataFormatID == 1)
            lasFile.publicHeader.pointDataRecordLength = 28;
         elseif (lasFile.publicHeader.pointDataFormatID == 3)
            lasFile.publicHeader.pointDataRecordLength = 34;
         end
      end
      
      function lasFile = set.color(lasFile, colorData)
         % Controlled interface for setting color data on LasFile objects.
         
         if (~strcmp(class(colorData),'uint16'))
            error('LasFile:DataTypeMismatch',...
               'Color data must be of type uint16');
         end
         
         if (size(colorData,1) ~= 3)
            error('LasFile:DimensionMismatch',...
               'Color data must have 3 coordinates');
         end
         
         if (size(colorData,2) ~= lasFile.publicHeader.nPoints)
            error('LasFile:DimensionMismatch',...
               'Color data must have an entry for each point');
         end
         
         lasFile.pColor = colorData;
         
         lasFile.publicHeader.pointDataFormatID = ...
            bitor(lasFile.publicHeader.pointDataFormatID,2);
         
         if (lasFile.publicHeader.pointDataFormatID == 2)
            lasFile.publicHeader.pointDataRecordLength = 26;
         elseif (lasFile.publicHeader.pointDataFormatID == 3)
            lasFile.publicHeader.pointDataRecordLength = 34;
         end
      end
      
      function lasFile = loadFrom(lasFile,location)
         % Load the LAS file from the given location.
         
         if (ischar(location))
            % Assume single argument is file name
            [fid, msg]=fopen(location, 'r');
            
            if (fid < 0)
               error('LasFile:FileError',msg);
            end
         elseif (isnumeric(location))
            % Assume single argument is file ID
            fid = location;
         else
            error('LasFile:InitError',...
               'Unknown argument initializer');
         end
         
         % Parse the LAS Public Header Block
         lasFile.publicHeader = LasPublicHeader(fid);
         
         % Parse the LAS Variable Length Record Headers
         if (lasFile.publicHeader.nVLRs > 0)
            vlrHeaders = LasVlr(fid);
            
            my_vlrs = cell(lasFile.publicHeader.nVLRs,1);
            
            for vlr=1:lasFile.publicHeader.nVLRs
               my_vlrs{vlr} = lasVlrFactory(vlrHeaders(vlr));
            end
            
            lasFile.vlrs = my_vlrs;
         end
         
         % Un-Pack data from uint8 for fast reading.
         fseek(fid, lasFile.publicHeader.offsetToPointData, 'bof');
         temp = reshape(fread(fid,...
            double(lasFile.publicHeader.pointDataRecordLength)*...
            lasFile.publicHeader.nPoints,...
            'uint8=>uint8'),...
            lasFile.publicHeader.pointDataRecordLength,...
            lasFile.publicHeader.nPoints);
         fclose(fid);
         
         offsets = diag([...
            lasFile.publicHeader.xOffset,...
            lasFile.publicHeader.yOffset,...
            lasFile.publicHeader.zOffset]);
         
         scales = diag([...
            lasFile.publicHeader.xScaleFactor,...
            lasFile.publicHeader.yScaleFactor,...
            lasFile.publicHeader.zScaleFactor]);
         
         tempPoints = temp(1:12,:);
         tempInt = temp(13:14,:);
         tempPtInfo = temp(15,:);
         tempPtClass = temp(16,:);
         tempSARank = temp(17,:);
         tempUData = temp(18,:);
         tempSrcID = temp(19:20,:);
         
         lasFile.pPoints = double(...
            reshape(typecast(tempPoints(:),'int32'),...
            3,lasFile.publicHeader.nPoints));
         
         lasFile.pPoints = scales*lasFile.pPoints+...
            offsets*ones(size(lasFile.pPoints));
         
         lasFile.intensity=reshape(typecast(tempInt(:),'uint16'),...
            1,lasFile.publicHeader.nPoints);
         
         lasFile.returnNumber=bitand(tempPtInfo,7);
         lasFile.numberOfReturns=bitand(bitshift(tempPtInfo,-3),7);
         lasFile.scanDirection=logical(bitget(tempPtInfo,7));
         lasFile.edgeOfFlightLine=logical(bitget(tempPtInfo,8));
         
         lasFile.withheld=logical(bitget(tempPtClass,8));
         lasFile.keyPoint=logical(bitget(tempPtClass,7));
         lasFile.synthetic=logical(bitget(tempPtClass,6));
         lasFile.classification=bitand(tempPtClass,31);
         
         lasFile.scanAngleRank=reshape(typecast(tempSARank(:),...
            'int8'),1,lasFile.publicHeader.nPoints);
         
         lasFile.userData=reshape(typecast(tempUData(:),'uint8'),...
            1,lasFile.publicHeader.nPoints);
         
         lasFile.pointSourceID=reshape(typecast(tempSrcID(:),...
            'uint16'),...
            1,lasFile.publicHeader.nPoints);
         
         switch lasFile.publicHeader.pointDataFormatID
            case 1
               tempGps = temp(21:28,:);
               
               lasFile.pGpsTime=reshape(typecast(tempGps(:),...
                  'double'),...
                  1,lasFile.publicHeader.nPoints);
            case 2
               tempColor = temp(21:26,:);
               
               lasFile.pColor = reshape(...
                  typecast(tempColor(:),'uint16'),...
                  3,lasFile.publicHeader.nPoints);
            case 3
               tempGps = temp(21:28,:);
               tempColor = temp(29:34,:);
               
               lasFile.pGpsTime=reshape(typecast(tempGps(:),...
                  'double'),...
                  1,lasFile.publicHeader.nPoints);
               
               lasFile.pColor = reshape(...
                  typecast(tempColor(:),'uint16'),...
                  3,lasFile.publicHeader.nPoints);
         end
         
         % Update offsets
         lasFile.publicHeader.offsetToPointData =...
            lasFile.publicHeader.headerSize;
         
         for record=1:lasFile.publicHeader.nVLRs
            lasFile.publicHeader.offsetToPointData = ...
               lasFile.publicHeader.offsetToPointData + 54 + ...
               lasFile.vlrs{record}.recordLengthAfterHeader;
         end
      end
      
      function saveTo(lasFile,location)
         % Save the LAS file to the given location.
         
         if (ischar(location))
            % Assume single argument is file name
            [fid, msg]=fopen(location, 'w');
            
            if (fid < 0)
               error('LasFile:FileError',msg);
            end
         elseif (isnumeric(location))
            % Assume single argument is file ID
            fid = location;
            
         else
            error('LasFile:InitError',...
               'Unknown argument initializer');
         end
         
         lasFile.publicHeader.saveTo(fid);
         
         for record=1:lasFile.publicHeader.nVLRs
            lasFile.vlrs{record}.saveTo(fid);
         end
         
         % Pack data to uint8 for fast writing.
         offs=diag([...
            lasFile.publicHeader.xOffset,...
            lasFile.publicHeader.yOffset,...
            lasFile.publicHeader.zOffset]);
         
         invSF=diag([...
            1/lasFile.publicHeader.xScaleFactor,...
            1/lasFile.publicHeader.yScaleFactor,...
            1/lasFile.publicHeader.zScaleFactor]);
         
         outPoints=int32(invSF*(lasFile.pPoints-...
            offs*ones(size(lasFile.pPoints))));
         
         uintPoints = reshape(typecast(outPoints(:),'uint8'),...
            12,lasFile.publicHeader.nPoints);
         
         uintInt = reshape(typecast(lasFile.intensity(:),'uint8'),...
            2,lasFile.publicHeader.nPoints);
         
         ptInfo=bitand(lasFile.returnNumber,7);
         ptInfo=bitor(bitshift(bitand(lasFile.numberOfReturns,7),3),ptInfo);
         ptInfo=bitset(ptInfo,7,lasFile.scanDirection);
         ptInfo=bitset(ptInfo,8,lasFile.edgeOfFlightLine);
         
         ptClass=bitand(lasFile.classification,31);
         ptClass=bitset(ptClass,6,lasFile.synthetic);
         ptClass=bitset(ptClass,7,lasFile.keyPoint);
         ptClass=bitset(ptClass,8,lasFile.withheld);
         
         uintSA = reshape(typecast(lasFile.scanAngleRank(:),'uint8'),...
            1,lasFile.publicHeader.nPoints);
         
         uintID = reshape(typecast(lasFile.pointSourceID(:),'uint8'),...
            2,lasFile.publicHeader.nPoints);
         
         switch lasFile.publicHeader.pointDataFormatID
            case 0
               temp = [uintPoints;...
                  uintInt;...
                  ptInfo;...
                  ptClass;...
                  uintSA;...
                  lasFile.userData;...
                  uintID];
            case 1
               uintGPS = reshape(typecast(lasFile.pGpsTime(:),'uint8'),...
                  8,lasFile.publicHeader.nPoints);
               
               temp = [uintPoints;...
                  uintInt;...
                  ptInfo;...
                  ptClass;...
                  uintSA;...
                  lasFile.userData;...
                  uintID;...
                  uintGPS];
            case 2
               uintColor = reshape(typecast(lasFile.pColor(:),'uint8'),...
                  6,lasFile.publicHeader.nPoints);
               
               temp = [uintPoints;...
                  uintInt;...
                  ptInfo;...
                  ptClass;...
                  uintSA;...
                  lasFile.userData;...
                  uintID;...
                  uintColor];
            case 3
               uintGPS = reshape(typecast(lasFile.pGpsTime(:),'uint8'),...
                  8,lasFile.publicHeader.nPoints);
               
               uintColor = reshape(typecast(lasFile.pColor(:),'uint8'),...
                  6,lasFile.publicHeader.nPoints);
               
               temp = [uintPoints;...
                  uintInt;...
                  ptInfo;...
                  ptClass;...
                  uintSA;...
                  lasFile.userData;...
                  uintID;...
                  uintGPS;...
                  uintColor];
         end
         
         fseek(fid, lasFile.publicHeader.offsetToPointData, 'bof');
         fwrite(fid,temp);
         
         if (ischar(location))
            % Close the file if it was opened  here.
            fclose(fid);
         end
      end
   end
end
