% LasVlr   ASPRS LAS format variable length record
%
% File:
%    LasVlr.m
%
% Description:
%    This MATLAB class represents an ASPRS LAS version 1.2 file variable length
%    record.
%
% Limitations:
%    None.
%
% Properties:
%    userID                   - The User Identifier
%    recordID                 - The Record Identifier
%    recordLengthAfterHeader  - The Record Length After Header
%    description              - The Description
%    data                     - The Data
%
% Methods:
%    [this] = LasVlr(varargin)  - Constructor for LAS VLR objects.
%             saveTo(location)  - Save LAS VLR to a given location.
%
% Other m-files required:
%    None.
%
% MAT-files required:
%    None.
%
% References:
%    http://asprs.org/a/society/committees/standards/asprs_las_format_v12.pdf
%
% See Also:
%    LasFile
%

%  Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

classdef LasVlr
   properties
      % The User Identifier
      %
      % The User ID field is ASCII character data that identifies the
      % user which created the variable length record. It is possible to
      % have many Variable Length Records from different sources with
      % different User IDs. If the character data is less than 16
      % characters, the remaining data must be null. The User ID must be
      % registered with the LAS specification managing body. The
      % management of these User IDs ensures that no two individuals
      % accidentally use the same User ID. The specification will
      % initially use two IDs: one for globally specified records
      % (LASF_Spec), and another for projection types (LASF_Projection).
      % Keys may be requested at
      % http://www.asprs.org/lasform/keyform.html.
      userID = ''
      
      % The Record Identifier
      %
      % The Record ID is dependent upon the User ID. There can be 0 to
      % 65535 Record IDs for every User ID. The LAS specification manages
      % its own Record IDs (User IDs owned by the specification),
      % otherwise Record IDs will be managed by the owner of the given
      % User ID. Thus each User ID is allowed to assign 0 to 65535 Record
      % IDs in any manner they desire. Publicizing the meaning of a given
      % Record ID is left to the owner of the given User ID. Unknown
      % User ID/Record ID combinations should be ignored.
      recordID = uint16(0)
      
      % The Record Length After Header
      %
      % The record length is the number of bytes for the record after the
      % end of the standard part of the header. Thus the entire record
      % length is 54 bytes (the header size iversion 1.2) plus the number
      % of bytes in the variable length portion of the record.
      recordLengthAfterHeader = uint16(0)
      
      % The Description
      %
      % Optional, null terminated description of the data. Any remaining
      % characters not used must be null.
      description = ''
      
      % The Data
      %
      % The raw VLR data as a byte array.
      data = zeros(0,'uint8');
   end
   
   properties (Dependent)
   end
   
   methods
      function this = LasVlr(varargin)
         % Constructor for LAS VLR objects
         
         switch (nargin)
            case 0
            case 1
               if (isa(varargin{1}, 'LasVlr'))
                  % Copy constructor
                  rhs = varargin{1};
                  fns = properties(rhs);
                  for i=1:length(fns)
                     this.(fns{i}) = rhs.(fns{i});
                  end
               else
                  if (ischar(varargin{1}))
                     % Assume single argument is file name
                     [fid, msg]=fopen(varargin{1}, 'rb');
                     
                     if (fid < 0)
                        error('LasVlr:FileError',msg);
                     end
                     
                     % Check the file signature to make sure we have a LAS file
                     if (~strcmp(sscanf(char(fread(fid,4,'uchar=>uchar')'),...
                                        '%c'),...
                                 'LASF'))
                        error('LasVlr:InvalidFile',...
                              'File does not appear to be a valid LAS file.');
                     end
                  elseif (isnumeric(varargin{1}))
                     % Assume single argument is file ID
                     fid = varargin{1};
                  else
                     error('LasVlr:InitError',...
                           ['Unknown argument initializer: ' varargin{1}]);
                  end
                  
                  fseek(fid, 94, 'bof');
                  headerSize = fread(fid,1,'uint16=>uint16');
                  
                  fseek(fid, 100, 'bof');
                  nVLRs = fread(fid,1,'uint16=>uint16');
                  
                  this(nVLRs) = LasVlr;
                  
                  fseek(fid,headerSize,'bof');
                  
                  for i=1:nVLRs
                     fread(fid,1,'uint16=>uint16');
                     this(i).userID = ...
                        sscanf(char(fread(fid,16,'uchar=>uchar')'),'%c');
                     this(i).recordID = ...
                        fread(fid,1,'uint16=>uint16');
                     this(i).recordLengthAfterHeader = ...
                        fread(fid,1,'uint16=>uint16');
                     this(i).description = ...
                        sscanf(char(fread(fid,32,'uchar=>uchar')'),'%c');
                     this(i).data = fread(fid,...
                        this(i).recordLengthAfterHeader,'uint8=>uint8');
                  end
                  
                  if (ischar(varargin{1}))
                     % Close the file if it was opened  here.
                     fclose(fid);
                  end
               end
            otherwise
               error('LasVlr:UnexpectedInputs',...
                     'Unexpected number of inputs encountered.');
         end
      end
      
      function saveTo(this, location)
         % Save the LAS file public header to the given location.
         
         if (ischar(location))
            % Assume single argument is file name
            [fid, msg]=fopen(location, 'w');
            
            if (fid < 0)
               error('LasPublicHeader:FileError',msg);
            end
         elseif (isnumeric(location))
            % Assume single argument is file ID
            fid = location;
         else
            error('LasPublicHeader:InitError',...
               'Unknown argument initializer');
         end
         
         fwrite(fid,0,'uint16');
         
         if (length(this.userID) > 16)
            fwrite(fid,this.userID(1:16),'char');
         else
            user=[this.userID...
               repmat(char(0),1,16-...
               numel(this.userID))];
            fwrite(fid,user,'char');
         end
         
         fwrite(fid,this.recordID,'uint16');
         fwrite(fid,this.recordLengthAfterHeader,'uint16');
         
         if (length(this.description) > 32)
            fwrite(fid,this.description(1:32),'char');
         else
            desc=[this.description...
               repmat(char(0),1,32-...
               numel(this.description))];
            fwrite(fid,desc,'char');
         end
         
         fwrite(fid,this.data,'uint8');
         
         if (ischar(location))
            % Close the file if it was opened  here.
            fclose(fid);
         end
      end
   end
end
