% LasfProjGeoKeyDirectoryTagVLR   LASF_Projection: GeoTIFF Tag - Key Directory
%
% File:
%    LasfProjGeoKeyDirectoryTagVLR.m
%
% Description:
%    This MATLAB class represents a LAS File Projection VLR containing the key
%    values that define the coordinate system. A complete description can be
%    found in the GeoTIFF format specification.
%
% Limitations:
%    None.
%
% Properties:
%    keyDirectoryVersion      - The Key Directory Version (1)
%    keyRevision              - The Key Revision (1)
%    minorRevision            - The Minor Revision (0)
%    numberOfKeys             - The Number of GeoTIFF Keys in this Directory
%    keys                     - The GeoTIFF Keys
%    userID                   - The User Identifier
%    recordID                 - The Record Identifier
%    recordLengthAfterHeader  - The Record Length After Header
%    description              - The Description
%    data                     - The Data
%
% Methods:
%    [this] = LasfProjGeoKeyDirectoryTagVLR(varargin)  - Constructor.
%             saveTo(location)                         - Save LAS VLR to a given
%                                                        location.
%
% Other m-files required:
%    LasVlr.m
%    geoKeyFactory.m
%
% MAT-files required:
%    None.
%
% References:
%    http://asprs.org/a/society/committees/standards/asprs_las_format_v12.pdf
%    http://www.remotesensing.org/geotiff/spec/geotiffhome.html
%
% See Also:
%    LasFile
%    LasVlr
%    geoKeyFactory
%

% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

classdef (Sealed = true) LasfProjGeoKeyDirectoryTagVLR < LasVlr
   properties
      keyDirectoryVersion = 1
      keyRevision = 1
      minorRevision = 0
      numberOfKeys = 0
      keys = {}
   end
   
   methods
      function vlr = LasfProjGeoKeyDirectoryTagVLR(varargin)
         if (nargin == 0)
            superargs = {};
         elseif (nargin == 1)
            if (~isa(varargin{1},'LasVlr'))
               error('LasfProjGeoKeyDirectoryTagVLR:InitError',...
                     'Unknown argument initializer.');
            end
            
            vlrHeader = varargin{1};
            
            superargs{1} = vlrHeader;
         else
            error('LasfProjGeoKeyDirectoryTagVLR:InitError',...
                  'Unexpected number of inputs.');
         end
         
         vlr = vlr@LasVlr(superargs{:});
         
         if (nargin == 1)
            
            if (~(strcmp(vlrHeader.userID, 'LASF_Projection') &&...
                  vlrHeader.recordID == 34735))
               error('LasfProjGeoKeyDirectoryTagVLR:InitError',...
                     'Invalid argument.');
            end
            
            % The GeoKeyDirectoryTag data is a stream of unsigned shorts
            data = typecast(vlrHeader.data,'uint16');
            
            if (data(1) ~= 1 && data(2) ~= 1 && data(3) ~= 0)
               error('LasfProjGeoKeyDirectoryTagVLR:InitError',...
                     'Invalid Format.');
            end
            
            vlr.numberOfKeys = data(4);
            
            keyData = reshape(data(5:end),4,vlr.numberOfKeys);

            vlr.keys = cell(vlr.numberOfKeys,1);
            
            for key=1:vlr.numberOfKeys
               tempKey=keyData(:,key);
               
               if (~(tempKey(1) == 0 && tempKey(2) == 0))
                  vlr.keys{key}=...
                     geoKeyFactory(tempKey(1),tempKey(2),tempKey(3),tempKey(4));
               end
            end
         end
      end
   end
end
