% LasfSpecClassificationVLR   LASF_Spec: Classification Dictionary
%
% File:
%    LasfSpecClassificationVLR.m
%
% Description:
%    This MATLAB class represents an ASPRS LAS version 1.2 file variable length
%    record.
%
% Limitations:
%    None.
%
% Properties:
%    classes                  - The Classification Dictionary
%    userID                   - The User Identifier
%    recordID                 - The Record Identifier
%    recordLengthAfterHeader  - The Record Length After Header
%    description              - The Description
%    data                     - The Data
%
% Methods:
%    [this] = LasfSpecClassificationVLR(varargin)  - Constructor.
%             saveTo(location)                     - Save LAS VLR to a given
%                                                    location.
%
% Other m-files required:
%    LasVlr.m
%
% MAT-files required:
%    None.
%
% References:
%    http://asprs.org/a/society/committees/standards/asprs_las_format_v12.pdf
%
% See Also:
%    LasFile
%    LasVlr
%

% Software History:
%    2012-AUG-29   K. Damkjer
%       Initial Coding.
%    2013-JUN-17   K. Damkjer
%       Additional Commenting.
%

classdef (Sealed = true) LasfSpecClassificationVLR < LasVlr
   properties (SetAccess = private)
      classes = containers.Map(...
                               { 0,...
                                 1,...
                                 2,...
                                 3,...
                                 4,...
                                 5,...
                                 6,...
                                 7,...
                                 8,...
                                 9,...
                                12},...
                               {'Not Classified',...
                                'Unclassified',...
                                'Ground',...
                                'Low Veg',...
                                'Medium Veg',...
                                'High Veg',...
                                'Building',...
                                'Noise',...
                                'Mass Point',...
                                'Water',...
                                'Overlap Point'});
   end
   
   methods
      function vlr = LasfSpecClassificationVLR(varargin)
         vlr = vlr@LasVlr(varargin{1});
         
         if (nargin == 1)
            if (~strcmp(class(varargin{1}),'uint8'))
               error('LasfSpecClassificationVLR:InitError',...
                     'Unknown argument initializer');
            end
            
            % The LasfSpecClassificationVLR data is a stream of uint8
            % char[15]s
            data = reshape(varargin{1},16,length(varargin{1})/16);
            
            keySet=num2cell(data(1,:));
            valueSet=cell(1,size(data,2));
            
            for i=1:size(data,2)
               valueSet{i}=sscanf(char(data(2:end,i)),'%c');
            end
            
            vlr.classes = containers.Map(keySet,valueSet,'UniformValues',true);
         end
      end
   end
end
