function [ elements ] = readPly( ply_file )
%%READPLY Read Stanford polygon (PLY) file.
%   Detailed explanation goes here
%

% Types
ply_types={'char','uchar','short','ushort','int','uint','float','double'...
'char8','uchar8','short16','ushort16','int32','uint32','float32','float64'};
mat_types={'schar','uchar','int16','uint16','int32','uint32','single','double'};

% Open the input file
[ fid, msg ] = fopen(ply_file, 'rt');

if (fid < 0)
   error(msg);
end

buffer = fgetl(fid);

magicnum = textscan(buffer, '%s');

if (numel(magicnum{1}) ~= 1 || ~strcmp(magicnum{1}{1}, 'ply'))
    fclose(fid);
    error('Not a PLY file.');
end

% Parse the header
format = 0;
celem = '';
elements = [];

while (true)
    % Read line
    buffer = fgetl(fid);
    key = '';
    pos = 0;

    % Tokenize
    if (~isempty(buffer))
        [tokens,pos] = textscan(buffer,'%s',1);
        key = tokens{1}{1};
    end

    switch lower(key)
        case 'format'
            tokens = textscan(buffer(pos+1:end), '%s %f');

            if (numel(tokens) ~= 2)
                fclose(fid);
                error(['Bad format definition: ', buffer]);
            end

            format  = lower(tokens{1}{1});
            version = tokens{2}(1);

            switch (format)
                case 'ascii'
                    format = 0;
                case 'binary_little_endian'
                    format = 1;
                case 'binary_big_endian'
                    format = 2;
                otherwise
                    fclose(fid);
                    error(['Format not supported: ' format]);
            end
            
            if (version ~= 1.0)
                fclose(fid);
                error('Only PLY version 1.0 supported.');
            end
            
        case 'comment'
            disp('Found Comment');
        case 'element'
            tokens = textscan(buffer(pos+1:end), '%s %u');

            if (numel(tokens) ~= 2)
                fclose(fid);
                error(['Bad element definition: ', buffer]);
            end

            name = tokens{1}{1};
            count = tokens{2}(1);

            if (isfield(elements,name))
                fclose(fid);
                error(['Duplicate element name: ', name]);
            end

            elements.(name).count = count;
            elements.(name).numProps = 0;
            elements.(name).hasListProp = false;
            elements.(name).allSameType = true;
            celem = name;
        case 'property'
            if (isempty(celem))
                fclose(fid);
                error(['Property definition without element: ' buffer]);
            end
            
            tokens = textscan(buffer(pos+1:end), '%s');

            if (numel(tokens{1}) < 2)
                fclose(fid);
                error(['Bad property definition: ', buffer]);
            end
            
            name=tokens{1}{numel(tokens{1})};
            
            if (isfield(elements.(celem),name))
                fclose(fid);
                error(['Duplicate property name: ', name]);
            end

            if (strcmpi(tokens{1}{1},'list'))
                if (numel(tokens{1}) ~= 4)
                    fclose(fid);
                    error(['Bad property definition: ', buffer]);
                end

                itype=strmatch(tokens{1}{2},ply_types,'exact');
                
                if (isempty(itype))
                    fclose(fid);
                    error(['Invalid data type: ' tokens{1}{2}]);
                end
                
                elements.(celem).hasListProp = true;

                elements.(celem).(name).list=mat_types{mod(itype-1,8)+1};

                itype=strmatch(tokens{1}{3},ply_types,'exact');
                
                if (isempty(itype))
                    fclose(fid);
                    error(['Invalid data type: ' tokens{1}{3}]);
                end

                elements.(celem).(name).type=mat_types{mod(itype-1,8)+1};
                elements.(celem).(name).values=cell(elements.(celem).count,1);

                propnames=fieldnames(elements.(celem));
                
                elements.(celem).allSameType =...
                    elements.(celem).allSameType &&...
                    strcmp(elements.(celem).(name).list,...
                           elements.(celem).(name).type) &&...
                    strcmp(elements.(celem).(propnames{5}).type,...
                           elements.(celem).(name).type);
            else
                if (numel(tokens{1}) ~= 2)
                    fclose(fid);
                    error(['Bad property definition: ', buffer]);
                end

                itype=strmatch(tokens{1}{1},ply_types,'exact');
                
                if (isempty(itype))
                    fclose(fid);
                    error(['Invalid data type: ' tokens{1}{1}]);
                end

                elements.(celem).(name).list='';
                elements.(celem).(name).type=mat_types{mod(itype-1,8)+1};
                elements.(celem).(name).values=zeros(elements.(celem).count,1);

                propnames=fieldnames(elements.(celem));
                
                elements.(celem).allSameType =...
                    elements.(celem).allSameType &&...
                    strcmp(elements.(celem).(propnames{5}).type,...
                           elements.(celem).(name).type);
            end
            
            elements.(celem).numProps = elements.(celem).numProps + 1;

        case 'end_header'
            % Save the location of the end of header in case we need to
            % re-load the file in binary mode.
            pos=ftell(fid);
            break;
        otherwise
    end
end

% Re-open as binary, if required
switch (format)
    case 0
        buffer=fscanf(fid,'%f');
        pos = 1.0;
        fclose(fid);
    case 1
        fclose(fid);
        fid = fopen(ply_file, 'rb', 'ieee-le.l64');
        fseek(fid,pos,'bof');
    case 2
        fclose(fid);
        fid = fopen(ply_file, 'rb', 'ieee-be.l64');
        fseek(fid,pos,'bof');
end

% Read Data

elemnames=fieldnames(elements);

if (format > 0)
    % Binary
    
    for ielem=1:length(elemnames)
        elemname=elemnames{ielem};
        element=elements.(elemname);
        propnames=fieldnames(element);

        msg=['Loading element ' num2str(ielem) ' of '...
             num2str(length(elemnames)) ' '];
        tstart=tic;
        h = timebar(1, element.count, msg, tstart);

        if (element.hasListProp)
            % Contains List Properties
            for elem = 1:element.count
                for iprop=5:length(propnames)
                    propname=propnames{iprop};
                    property=element.(propname);
                    
                    if (isempty(property.list))
                        elements.(elemname).(propname).values(elem)=...
                            fread(fid,1,property.type);
                    else
                        items=fread(fid,1,property.list);
                        
                        if (items > 0)
                            elements.(elemname).(propname).values{elem}=...
                                fread(fid,[1 items],property.type);
                        end
                    end
                end
                
                if (toc > 1)
                    tic;
                    h = timebar(elem, element.count, msg, tstart, h);
                end
            end
        else
            % Scalar Properties Only
            if (element.allSameType)
                % Homogeneous Types
                data=fread(fid,...
                           [element.numProps,element.count],...
                           element.(propnames{5}).type)';

                for iprop=5:length(propnames)
                    propname=propnames{iprop};
                    elements.(elemname).(propname).values=data(:,iprop-4);
                end
            else
                % Non-Homogeneous Types
                for elem = 1:element.count
                    for iprop=5:length(propnames)
                        propname=propnames{iprop};
                        property=element.(propname);
                        elements.(elemname).(propname).values(elem)=...
                            fread(fid,1,property.type);
                    end
                    
                    if (toc > 1)
                        tic;
                        h = timebar(elem, element.count, msg, tstart, h);
                    end
                end
            end
        end
            
        if (all(ishghandle(h, 'figure')))
            close(h);
        end
    end
    
    fclose(fid);
else
    % ASCII

    for ielem=1:length(elemnames)
        elemname=elemnames{ielem};
        element=elements.(elemname);
        propnames=fieldnames(element);

        msg=['Loading element ' num2str(ielem) ' of '...
             num2str(length(elemnames)) ' '];
        tstart=tic;
        h = timebar(1, length(elemnames), msg, tstart);
        tic;

        if (element.hasListProp)
            % Parse element with list properties
            for elem=1:element.count
                for iprop=5:length(propnames)
                    propname=propnames{iprop};
                    property=element.(propname);
                
                    if (isempty(property.list))
                        elements.(elemname).(propname).values(elem)=...
                            buffer(pos);
                    else
                        items=typecast(buffer(pos),class(pos));

                        if (items > 0)
                            elements.(elemname).(propname).values{elem}=...
                                buffer(pos+(1:items))';
                            pos = pos + items;
                        end
                    end
                    
                    pos = pos + 1;
                end
                
                if (toc > 1)
                    tic;
                    h = timebar(elem, element.count, msg, tstart, h);
                end
            end
        else
            % Fast parse element without list properties
            data=reshape(buffer(pos:pos+element.count*element.numProps-1),...
                         element.numProps, element.count)';
            pos=pos+element.count*element.numProps;
            
            for iprop=5:length(propnames)
                propname=propnames{iprop};
                elements.(elemname).(propname).values=data(:,iprop-4);
            end

            clear data;
        end
            
        if (all(ishghandle(h, 'figure')))
            close(h);
        end
    end
end

end
