//*************************************************************************
// FILE:        ClassHandle.h
//
//    Copyright (C)  2012 Kristian L. Damkjer.
//
// DESCRIPTION: This class is intended to be used to wrap C++ objects for
//              transport across mex function calls in MATLAB.
//
// LIMITATIONS: To preserve the lifetime guarantees of the pointer, default
//              construction, copy construction and assignment are
//              explicitly disallowed.
//
//              The class handle owns the pointer it contains. Thus, when
//              the class handle is destroyed, the pointer is deleted.
//
//              Class handles are forced to be allocated on the heap by
//              using the named constructor idiom. Class handles should be
//              destroyed using the destroyHandleTo template function.
//
// SOFTWARE HISTORY:
//> 2012-OCT-08  K. Damkjer
//               Initial Coding.
//<
//*************************************************************************
#ifndef ClassHandle_HEADER
#define ClassHandle_HEADER

#ifndef EXPORT
   #if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
      #define EXPORT
   #else
      #define EXPORT export
   #endif
#endif

#include <string>

#ifdef _CHAR16T
#define CHAR16_T
#endif

#include "mex.h"

template<typename BASE>
mxArray* ptrAsMat(BASE*);

template<typename BASE>
class ClassHandle
{
public:
   ~ClassHandle();

   const BASE& obj() const { return *thePointer; }
   
   bool isValid() const;

   friend mxArray* ptrAsMat<BASE>(BASE*);

private:
   // We want to force objects to be created using the ptrAsMat named
   // constructor idiom. This also forces ClassHandles to be allocated on
   // the heap instead of on the stack.
   ClassHandle(BASE*);

   // Explicitly disable "default" constructors.
   ClassHandle();
   ClassHandle(const ClassHandle<BASE>&);
   ClassHandle<BASE>& operator=(const ClassHandle<BASE>&);

   BASE* thePointer;
   std::string theName;
   uint32_T theSignature;
};

//*************************************************************************
// Function: ptrAsMat
//*************************************************************************
template<typename BASE>
mxArray* ptrAsMat(BASE* ptr)
{
   mxArray* mat = mxCreateNumericMatrix(1, 1, mxINDEX_CLASS, mxREAL);
   *((mwIndex*)mxGetData(mat)) =
                     reinterpret_cast<mwIndex>(new ClassHandle<BASE>(ptr));
   return mat;
}

//*************************************************************************
// FUNCTION: matAsObj
//*************************************************************************
template<typename BASE>
const BASE& matAsObj(const mxArray* mat)
{
   if (mxGetNumberOfElements(mat) != 1    ||
       mxGetClassID(mat) != mxINDEX_CLASS ||
       mxIsComplex(mat))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsPtr:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }
   
   ClassHandle<BASE>* handle =
         reinterpret_cast<ClassHandle<BASE>*>(*((mwIndex*)mxGetData(mat)));

   if (!(handle->isValid()))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsPtr:invalidHandle",
                        "Handle not valid.");
   }

   return handle->obj();
}

//*************************************************************************
// FUNCTION: destroyHandleTo
//*************************************************************************
template<typename BASE>
void destroyHandleTo(const mxArray* mat)
{
   if (mxGetNumberOfElements(mat) != 1    ||
       mxGetClassID(mat) != mxINDEX_CLASS ||
       mxIsComplex(mat))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsPtr:invalidHandle",
                        "Input must be real-valued index-class scalar.");
   }
   
   ClassHandle<BASE>* handle =
         reinterpret_cast<ClassHandle<BASE>*>(*((mwIndex*)mxGetData(mat)));

   if (!(handle->isValid()))
   {
      mexErrMsgIdAndTxt("Damkjer:matAsPtr:invalidHandle",
                        "Handle not valid.");
   }

   delete handle;
}

#if defined(MAKE_DEPENDING) || defined(TEMPLATE_INCLUSION)
#include "ClassHandle.cpp"
#endif

#endif
