%%
% [COLORS] = NORMCOLOR(QUERIES, DATABASE, ...)
%
% Color points by surface normal estimate orientation.
%
% Software History:
%    2013-JAN-29  Damkjer, K.
%       Updated Error Statement to reflect appropriate package.
%
function colors=normcolor(QUERIES, DATABASE, varargin)

% Parse Inputs
userParams = parseInputs(varargin{:});

norms=zeros(size(QUERIES));

% dimensions=size(QUERIES, 1);
elements=size(QUERIES, 2);

msg='Computing Normal Colors...';
tstart=tic;
h = timebar(1, elements, msg, tstart);

step=1000;
tic;
for elem=1:step:elements
    
   last=min(elem+step-1,elements);
    
   % Get the nearest neighbors of elem
   if (userParams.neighbors <= 0)
       % Perform a fixed radius search
       NN=DATABASE.rnn(QUERIES(:,elem:last),...
                       userParams.radius);
   else
       if (userParams.radius <= 0)
           % Search unconstrained
           NN=DATABASE.knn(QUERIES(:,elem:last),...
                           userParams.neighbors);
       else
           % Search constrained to radius
           NN=DATABASE.knn(QUERIES(:,elem:last),...
                           userParams.neighbors,...
                           'lim',userParams.radius);
       end
   end

   covs=fastcov(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));
%    recentered=fastcenter(cellfun(@(x) QUERIES(:,x)',NN,'UniformOutput',false));

   for nbr=1:size(NN,1)
      % skip underconstrained (possible with radius searches)
      if (length(NN{nbr})<5)
         continue;
      end
      
% %       [~,S,V]=svd(covs{nbr},0);
% %       norms(:,elem+nbr-1)=abs(V(:,dimensions) .* S(dimensions, dimensions));
% 
%       [~,~,V]=svd(recentered{nbr},0);
% %       S = S ./ sqrt(length(NN{nbr})-1);
% %       norms(:,elem+nbr-1)=abs(V(:,dimensions) .* S(dimensions, dimensions));

      [V,D]=eig(covs{nbr});
      [~,index]=min(abs(diag(D)));
      norms(:,elem+nbr-1)=abs(V(:,index));
%       norms(:,elem+nbr-1)=abs(V(:,dimensions));
   end
   
   if (toc > 1)
      tic;
      h = timebar(elem, elements, msg, tstart, h);
   end
end

if (all(ishghandle(h, 'figure')))
   close(h);
end

colors=uint16(double(intmax('uint16'))*norms/max(max(norms)));

end

%%
% PARSEINPUTS    Support function to parse inputs into userParams structure
function [userParams] = parseInputs(varargin)

userParams = struct('radius', 0, 'neighbors', 0);

if length(varargin) == 1 || ~isnumeric(varargin{2})
    value = varargin{1};

    if (isscalar(value)  && ...
        isreal(value)    && ...
        value >= 5)
       userParams.neighbors = fix(value);
    else
       error('Damkjer:InvalidCount', ...
             ['Number of Neighbors must be a real valued positive '...
              'integer greater or equal to 5: ' num2str(value)]);
    end

    varargin(1) = [];
end

% Parse the Property/Value pairs
if rem(length(varargin), 2) ~= 0
   error('Damkjer:PropertyValueNotPair', ...
         ['Additional arguments must take the form of Property/Value '...
          'pairs']);
end

propertyNames = {'neighbors', 'radius'};

while ~isempty(varargin)
   property = varargin{1};
   value    = varargin{2};

   % If the property has been supplied in a shortened form, lengthen it
   iProperty = find(strncmpi(property, propertyNames, length(property)));

   if isempty(iProperty)
      error('Damkjer:InvalidProperty', 'Invalid Property');
   elseif length(iProperty) > 1
      error('Damkjer:AmbiguousProperty', ...
            'Supplied shortened property name is ambiguous');
   end
   
   property = propertyNames{iProperty};

   switch property
      case 'neighbors'
         if (isscalar(value)  && ...
             isreal(value)    && ...
             value >= 5)
            userParams.neighbors = fix(value);
         else
       error('Damkjer:InvalidCount', ...
             ['Number of Neighbors must be a real valued positive '...
              'integer greater or equal to 5']);
         end
      case 'radius'
         if (isscalar(value) && ...
             isnumeric(value) && ...
             isreal(value) && ...
             value > 0)
            userParams.radius = value;
         else
            error('Damkjer:InvalidPointWeights', ...
                  'Radius must be a real valued positive scalar');
         end
   end

    varargin(1:2) = [];
end

if (userParams.neighbors <= 0 && userParams.radius <= 0)
   userParams.radius = 1;
end

end
